#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <wx/wx.h>
#include <wx/spinctrl.h>
#include <wx/dc.h>
#include <wx/colordlg.h>

#include "painter.h"
#include "controls.h"
#include "guibuiltin.h"
#include "xcsguiview.h"
#include "xcspropedit.h"
#include "editctrls.h"


#define ITEM_HEIGHT 23
#define LABEL_WIDTH 67

BEGIN_EVENT_TABLE( XcsPropEdit, wxWindow )
	EVT_LEFT_DOWN( XcsPropEdit::OnLeftButtonDown )
	EVT_LEFT_DCLICK( XcsPropEdit::OnLeftDblClick )
	EVT_LEFT_UP( XcsPropEdit::OnLeftButtonUp )
	EVT_RIGHT_DOWN( XcsPropEdit::OnRightButtonDown )
	EVT_MOTION( XcsPropEdit::OnMouseMove )
	EVT_CHAR( XcsPropEdit::OnKeyEvent )
	EVT_PAINT( XcsPropEdit::OnPaint )
	EVT_SIZE( XcsPropEdit::OnSize )
END_EVENT_TABLE()


XcsPropEdit::XcsPropEdit(wxWindow *parent, XcsGUIView *view)
	: wxWindow(parent, -1, wxPoint(100,100), wxSize(170, 200), wxWANTS_CHARS)
{
	mGUIView = view;
	mCurrentObj = NULL;
	mBufBitmap = new wxBitmap(300, 300);

	int i;
	for (i=0;i<MAX_EDIT_PROPS;i++)
	{
		mPInfo[i].pedit = this;
		mPInfo[i].prop = NULL;
		mPInfo[i].name = "no name";
		mPInfo[i].type = -1;
		mPInfo[i].ctrl = NULL;
		mPInfo[i].updating = false;
	}

	nProps = 0;
	mCurrentObj = NULL;

}

XcsPropEdit::~XcsPropEdit()
{
	/* wxGrid destructor will delete the table */
	delete mBufBitmap;

	for (int i=0;i<MAX_EDIT_PROPS;i++)
	{
		if (mPInfo[i].ctrl != NULL)
			delete mPInfo[i].ctrl;
	}
}

static void cb_entry_changed(entry *e, void *data)
{
	pinfo *inf = (pinfo*)data;
	bool dosync = false;
	wxString str;

	if (inf->updating)
		return;

	switch(inf->type)
	{
	case PROP_INTEGER:
		inf->prop->ivalue = e->getint();
		dosync = true;
		break;
	case PROP_FLOAT:
		inf->prop->fvalue = e->getfloat();
		dosync = true;
		break;
	case PROP_STRING:
		str = e->gettext();
		inf->prop->string = str;
		dosync = true;
		break;
	default:
		break;
	}

	if (dosync)
	{
		inf->pedit->GetGUIView()->SetDirty();

		if (inf->prop->sync_cb)
			(*(inf->prop->sync_cb))(inf->prop);
	}
}

static void cb_checkedit_changed(checkedit *e, void *data)
{
	pinfo *inf = (pinfo*)data;
	if (inf->updating)
		return;

	inf->pedit->GetGUIView()->SetDirty();
	inf->prop->bvalue = e->getvalue();
	if (inf->prop->sync_cb)
		(*(inf->prop->sync_cb))(inf->prop);

}

static void cb_strlistedit_changed(strlistedit *e, void *data)
{
	pinfo *inf = (pinfo*)data;
	if (inf->updating)
		return;

	inf->pedit->GetGUIView()->SetDirty();
	inf->prop->string_array = e->getstrings();
	if (inf->prop->sync_cb)
		(*(inf->prop->sync_cb))(inf->prop);
}

static void cb_colouredit_changed(colouredit *e, void *data)
{
	pinfo *inf = (pinfo*)data;
	if (inf->updating)
		return;

	inf->pedit->GetGUIView()->SetDirty();
	inf->prop->colour = e->getcolour();
	if (inf->prop->sync_cb)
		(*(inf->prop->sync_cb))(inf->prop);
}

XcsGUIView *XcsPropEdit::GetGUIView()
{
	return mGUIView;
}

void XcsPropEdit::SetObject(Object *obj)
{
	if (obj == mCurrentObj)
		return;

	mCurrentObj = obj;

	
	for (int i=0;i<MAX_EDIT_PROPS;i++)
	{
		if (mPInfo[i].ctrl != NULL)
		{
			delete mPInfo[i].ctrl;
			mPInfo[i].ctrl = NULL;
		}
	}

	if (mCurrentObj == NULL)
	{
		nProps = 0;
	}
	else
	{	
		nProps = mCurrentObj->NumProps();
		int i;
		for (i=0;i<nProps && i<MAX_EDIT_PROPS;i++)
		{
			mPInfo[i].prop = mCurrentObj->GetProp(i);
			mPInfo[i].type = mPInfo[i].prop->type;

			wxString name = mPInfo[i].prop->name;
			if (name.Len() > 0)
				name.GetWritableChar(0) = toupper(name[0]);
			mPInfo[i].name = name;

			
			switch(mPInfo[i].type)
			{
			case PROP_INTEGER:
				mPInfo[i].ctrl = new entry(this);
				((entry*)mPInfo[i].ctrl)->setmode(entry::INTEGER);
				((entry*)mPInfo[i].ctrl)->changedcallback( cb_entry_changed, &mPInfo[i] );
				if (mPInfo[i].prop->min != -999)
					((entry*)mPInfo[i].ctrl)->setrange(mPInfo[i].prop->min, mPInfo[i].prop->max);
				if (mPInfo[i].prop->step != -999)
					((entry*)mPInfo[i].ctrl)->setstep(mPInfo[i].prop->step);
				break;
			case PROP_FLOAT:
				mPInfo[i].ctrl = new entry(this);
				((entry*)mPInfo[i].ctrl)->setmode(entry::FLOAT);
				((entry*)mPInfo[i].ctrl)->changedcallback( cb_entry_changed, &mPInfo[i] );
				break;
			case PROP_STRING:
				mPInfo[i].ctrl = new entry(this);
				((entry*)mPInfo[i].ctrl)->setmode(entry::STRING);
				((entry*)mPInfo[i].ctrl)->changedcallback( cb_entry_changed, &mPInfo[i] );
				break;
			case PROP_BOOLEAN:
				mPInfo[i].ctrl = new checkedit(this);
				((checkedit*)mPInfo[i].ctrl)->togglecallback( cb_checkedit_changed, &mPInfo[i] );
				break;
			case PROP_STRING_ARRAY:
				mPInfo[i].ctrl = new strlistedit(this);
				((strlistedit*)mPInfo[i].ctrl)->changedcallback( cb_strlistedit_changed, &mPInfo[i]);
				break;
			case PROP_COLOUR:
				mPInfo[i].ctrl = new colouredit(this);
				((colouredit*)mPInfo[i].ctrl)->changedcallback( cb_colouredit_changed, &mPInfo[i]);
				break;
			case PROP_MENU:
			default:
				mPInfo[i].ctrl = NULL;
				break;
			}
			mPInfo[i].updating = true;
			PropertyToView(mPInfo[i]);
			mPInfo[i].updating = false;
		}
	}		

	DoLayout();
	RedrawAll();

}

void XcsPropEdit::UpdateView()
{
	for (int i=0;i<nProps;i++)
	{
		mPInfo[i].updating = true;
		PropertyToView(mPInfo[i]);
		mPInfo[i].updating = false;
	}
	RedrawAll();
}

void XcsPropEdit::PropertyToView(const pinfo &inf)
{
	if (inf.ctrl != NULL)
	{
		switch(inf.type)
		{
		case PROP_INTEGER:
			((entry*)inf.ctrl)->setval( (int)inf.prop->ivalue );
			break;
		case PROP_FLOAT:
			((entry*)inf.ctrl)->setval( (float)inf.prop->fvalue );
			break;
		case PROP_STRING:
			((entry*)inf.ctrl)->settext( inf.prop->string.c_str() );
			break;
		case PROP_BOOLEAN:
			((checkedit*)inf.ctrl)->setvalue( inf.prop->bvalue );
			break;
		case PROP_STRING_ARRAY:
			((strlistedit*)inf.ctrl)->setstrings( inf.prop->string_array );
			break;
		case PROP_COLOUR:
			((colouredit*)inf.ctrl)->setcolour( inf.prop->colour );
			break;
		case PROP_MENU:
		default:
			break;
		}
	}
}

void XcsPropEdit::RedrawAll()
{
	wxMemoryDC dc;
	dc.SelectObject(*mBufBitmap);
	
	Painter gfx(&dc);
	Draw(dc);

	wxClientDC cdc(this);
	cdc.Blit(0, 0, mBufBitmap->GetWidth(), mBufBitmap->GetHeight(), &dc, 0, 0);

	dc.SelectObject( wxNullBitmap );
}


void XcsPropEdit::DoLayout()
{
	wxSize sz = GetSize();
	for (int i=0;i<nProps;i++)
	{
		mPInfo[i].labelarea = wxRect(3, i*ITEM_HEIGHT, LABEL_WIDTH-4, ITEM_HEIGHT);
		mPInfo[i].ctrlarea = wxRect(LABEL_WIDTH+1, i*ITEM_HEIGHT+1, sz.GetWidth()-LABEL_WIDTH-1, ITEM_HEIGHT-2);
		if (mPInfo[i].ctrl != NULL)
			mPInfo[i].ctrl->setgeom( mPInfo[i].ctrlarea );
	}
}

void XcsPropEdit::Draw(wxDC &dc)
{
	Painter gfx(&dc);
	wxSize sz = GetSize();
	gfx.SetSysControlColour();
	gfx.DrawRectangle(0, 0, LABEL_WIDTH, sz.GetHeight());
	gfx.SetColour( *wxWHITE );
	gfx.DrawRectangle(LABEL_WIDTH, 0, sz.GetWidth()-LABEL_WIDTH, sz.GetHeight());	

	gfx.SetColour( *wxBLACK );
	gfx.SetHelveticaTinyBold();


	for (int i=0;i<nProps;i++)
	{
		gfx.GetDC()->SetClippingRegion(mPInfo[i].labelarea);
		gfx.DrawString(5, i*ITEM_HEIGHT + ITEM_HEIGHT/2-gfx.GetCharHeight()/2, mPInfo[i].name);
		gfx.GetDC()->DestroyClippingRegion();

		gfx.DrawLine(0, i*ITEM_HEIGHT, sz.GetWidth(), i*ITEM_HEIGHT);
	}
	gfx.DrawLine(0, i*ITEM_HEIGHT, sz.GetWidth(), i*ITEM_HEIGHT); // draw the last line too


	/* now render the property value editors */
	gfx.SetHelveticaSmall();

	for (i=0;i<nProps;i++)
	{

		gfx.SetDefaults();
		gfx.GetDC()->SetClippingRegion(mPInfo[i].ctrlarea);
		if (mPInfo[i].ctrl != NULL)
		{
			mPInfo[i].ctrl->draw(gfx);
		}
		gfx.GetDC()->DestroyClippingRegion();
	}

}

void XcsPropEdit::OnLeftButtonDown(wxMouseEvent &evt)
{
	CaptureMouse();
	int i;
	for (i=0;i<MAX_EDIT_PROPS;i++)
	{
		if (mPInfo[i].ctrl != NULL)
			mPInfo[i].ctrl->setfocus(false);
	}

	for (i=0;i<nProps;i++)
	{
		int x = evt.GetX();
		int y = evt.GetY();
		if (mPInfo[i].ctrlarea.Inside(x, y))
		{
			if (mPInfo[i].ctrl != NULL)
			{
				mPInfo[i].ctrl->setfocus(true);
				this->SetFocus();
				mPInfo[i].ctrl->onleftdown(evt);
			}
		}
	}
}

void XcsPropEdit::OnLeftButtonUp(wxMouseEvent &evt)
{
	int i;
	for (i=0;i<nProps;i++)
	{
		if (mPInfo[i].ctrl != NULL && mPInfo[i].ctrl->hasfocus())
			mPInfo[i].ctrl->onleftup(evt);
	}

	if (HasCapture())
		ReleaseMouse();
}

void XcsPropEdit::OnLeftDblClick(wxMouseEvent &evt)
{
	int i;
	for (i=0;i<MAX_EDIT_PROPS;i++)
	{
		if (mPInfo[i].ctrl != NULL)
			mPInfo[i].ctrl->setfocus(false);
	}

	for (i=0;i<nProps;i++)
	{
		int x = evt.GetX();
		int y = evt.GetY();
		if (mPInfo[i].ctrlarea.Inside(x, y))
		{
			if (mPInfo[i].ctrl != NULL)
			{
				mPInfo[i].ctrl->setfocus(true);
				this->SetFocus();
				mPInfo[i].ctrl->ondblclick(evt);
			}
		}
	}
}

void XcsPropEdit::OnRightButtonDown(wxMouseEvent &evt)
{
}

void XcsPropEdit::OnKeyEvent(wxKeyEvent &evt)
{
	int i;
	for (i=0;i<nProps;i++)
	{
		if (mPInfo[i].ctrl != NULL && mPInfo[i].ctrl->hasfocus())
			mPInfo[i].ctrl->onchar(evt);
	}
}

void XcsPropEdit::OnMouseMove(wxMouseEvent &evt)
{
	int i;
	for (i=0;i<nProps;i++)
	{
		int x = evt.GetX();
		int y = evt.GetY();
		if (mPInfo[i].ctrlarea.Inside(x, y))
		{
			if (mPInfo[i].ctrl != NULL)
				mPInfo[i].ctrl->onmousemove(evt);
		}
	}
}


void XcsPropEdit::OnSize(wxSizeEvent &evt)
{
	wxSize wsz = evt.GetSize();
	if (mBufBitmap->GetWidth() < wsz.GetWidth() ||
		mBufBitmap->GetHeight() < wsz.GetHeight() )
	{
		delete mBufBitmap;
		mBufBitmap = new wxBitmap(wsz.GetWidth(), wsz.GetHeight());
		RedrawAll();
	}
}

void XcsPropEdit::OnPaint(wxPaintEvent &evt)
{
	wxPaintDC dc(this);
	PrepareDC(dc);

	dc.BeginDrawing();
	Draw( dc );
	dc.EndDrawing();

}


/*
		wxRect rct = mPInfo[i].ctrlarea;
		int texty = i*ITEM_HEIGHT + ITEM_HEIGHT/2-gfx.GetCharHeight()/2;


		if (mPInfo[i].type == PROP_INTEGER)
		{
			wxString str = wxString::Format("%d", mPInfo[i].prop->ivalue);
			gfx.DrawString(rct.x+rct.width-3-gfx.GetTextWidth(str), texty, str);
		}
		else if (mPInfo[i].type == PROP_FLOAT)
		{
			wxString str = wxString::Format("%.3f", mPInfo[i].prop->fvalue);
			gfx.DrawString(rct.x+rct.width-3-gfx.GetTextWidth(str), texty, str);
		}
		else if (mPInfo[i].type == PROP_BOOLEAN)
		{
			wxRect chkrect(rct.x+3, i*ITEM_HEIGHT+4, ITEM_HEIGHT-6, ITEM_HEIGHT-6);
			if (mPInfo[i].prop->bvalue)
				gfx.DrawCheckMark(chkrect.x, chkrect.y, chkrect.width, chkrect.height, 3);
			gfx.DrawRectangle(chkrect.x, chkrect.y, chkrect.width, chkrect.height, false);
		}
		else if (mPInfo[i].type == PROP_STRING)
		{		
			gfx.DrawString(rct.x+3, texty, mPInfo[i].prop->string);
		}
		else if (mPInfo[i].type == PROP_STRING_ARRAY ||
				mPInfo[i].type == PROP_MENU )
		{
			gfx.DrawString(rct.x+4, texty, "...");
		}
		else if (mPInfo[i].type == PROP_COLOUR)
		{
			gfx.SetColour( mPInfo[i].prop->colour );
			gfx.DrawRaisedPanel( rct.x+3, rct.y+4, rct.height-6, rct.height-6);
		}
		else
		{
			wxFAIL_MSG("unsupported property type in property editor");
		}
*/