#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <wx/wx.h>
#include <wx/scrolbar.h>
#include <wx/print.h>
#include <wx/printdlg.h>
#include <wx/accel.h>

#include "guibuiltin.h"
#include "xcsguiview.h"
#include "xcspropedit.h"
#include "xcshelp.h"
#include "xcswx.h"


#define RSZBOXW 6
#define BOX_NONE 0
#define BOX_TOPLEFT 1
#define BOX_TOPRIGHT 2
#define BOX_BOTTOMLEFT 3
#define BOX_BOTTOMRIGHT 4
#define BOX_TOP 5
#define BOX_LEFT 6
#define BOX_RIGHT 7
#define BOX_BOTTOM 8

#define TITLE_XOFF 5
#define FRAME_WIDTH 5
#define FRAME_HEIGHT 25

ActionInfo::ActionInfo(ActionType type)
{
	this->type = type;
}

ActionInfo::~ActionInfo()
{
	/* nothing to do */
}

BEGIN_EVENT_TABLE( XcsGUIView, wxWindow )
	EVT_LEFT_DOWN( XcsGUIView::OnLeftButtonDown )
	EVT_LEFT_UP( XcsGUIView::OnLeftButtonUp )
	EVT_RIGHT_DOWN( XcsGUIView::OnRightButtonDown )
	EVT_MOTION( XcsGUIView::OnMouseMove )
	EVT_CHAR( XcsGUIView::OnKeyEvent )
	EVT_PAINT( XcsGUIView::OnPaint )
	EVT_SIZE( XcsGUIView::OnSize )
END_EVENT_TABLE()

XcsGUIView::XcsGUIView(XcsFrame *frm)
	: wxWindow(frm, -1, wxPoint(10, 10), wxSize(200, 200), wxWANTS_CHARS)
{
	mFrame = frm;
	mPropEdit = NULL;
		
	mCurrentDialog = NULL;
	mCurrentActionInfo = NULL;

	mSelectColour = "magenta";
	mForeground = "black";
	mBackground = "white";
	mFrameColour = wxSystemSettings::GetColour(wxSYS_COLOUR_ACTIVECAPTION);

	SetBackgroundColour( mBackground );

	mStandardCursor = wxCursor( wxCURSOR_ARROW ); 
	mMoveResizeCursor = wxCursor( wxCURSOR_SIZING );
	mNwSeCursor = wxCursor( wxCURSOR_SIZENWSE );
	mNSCursor = wxCursor( wxCURSOR_SIZENS );
	mWECursor = wxCursor( wxCURSOR_SIZEWE );
	mNeSwCursor = wxCursor( wxCURSOR_SIZENESW );

	SetCursor( mStandardCursor );
	
	bDirty = false;
	bMoveMode = false;
	bMoveModeErase = false;
	mOrigX = mOrigY = mDiffX = mDiffY = 0;

	bResizeMode = false;
	bResizeModeErase = false;
	mResizeBox = BOX_NONE;

	mBufBitmap = new wxBitmap(300, 400);

	bDedCompat = false;
}

XcsGUIView::~XcsGUIView()
{
	FreeUndoStack();
	FreeRedoStack();

	DeleteDialogs();

	delete mBufBitmap;
}

void XcsGUIView::SetDirty(bool d)
{
	bDirty = d;
}

bool XcsGUIView::IsDirty()
{
	return bDirty;
}

XcsFrame *XcsGUIView::GetXcsFrame()
{
	return mFrame;
}

void XcsGUIView::SetPropEditor(XcsPropEdit *pedit)
{
	mPropEdit = pedit;
}

Dialog *XcsGUIView::MakeNewDialog(const wxString &name)
{
	Dialog *fm1 = new Dialog(this, name);
	AddDialog(fm1);
	SetActiveDialog(fm1);
	fm1->SetGeom(wxRect(15, 35, 400, 300));
	return fm1;
}

void XcsGUIView::CreateNewControl(ControlInfo *ctrlinfo)
{
	wxString name;
	bool name_ok = false;
	int index = 1;
	if (ctrlinfo->type == CTRL_DIALOG)
	{
		// create a new dialog
		while (name_ok == false)
		{
			name.Printf("%s%d", ctrlinfo->name, index++);
			if (FindDialog(name) == NULL)
				name_ok = true;
		}

		Dialog *fm1 = MakeNewDialog( name );
		RedrawAll();
		mFrame->GetDlgSelector()->UpdateList();
		mFrame->GetDlgSelector()->SetDialog( fm1 );
		bDirty = true;
		mFrame->UpdateControlStatus();
	}
	else if (mCurrentDialog != NULL)
	{
		// create a new control
		while (!name_ok)
		{
			name.Printf("%s%d", ctrlinfo->name, index++);
			if (mCurrentDialog->FindChild(name) == NULL)
				name_ok = true;
		}

		Object *obj1 = new Object( ctrlinfo, name );
		obj1->SetDialog( mCurrentDialog );
		mSelectedItems.clear();
		RedrawAll();
		mFrame->UpdateControlStatus();
		bDirty = true;
	}
	else
	{
		wxMessageBox("Sorry, could not create the new control.", "Error", wxOK | wxICON_ERROR);
	}
}

void XcsGUIView::AddDialog(Dialog *frm)
{
	mDialogs.append(frm);
}

Dialog *XcsGUIView::FindDialog(const wxString &name)
{
	int ct = mDialogs.count();
	for (int i=0;i<ct;i++)
		if (mDialogs[i]->GetName() == name)
			return mDialogs[i];
	
	return NULL;
}


Dialog *XcsGUIView::FindDialogByTitle(const wxString &title)
{
	int ct = mDialogs.count();
	for (int i=0;i<ct;i++)
		if (mDialogs[i]->GetTitle() == title)
			return mDialogs[i];
	
	return NULL;
}

void XcsGUIView::RemoveDialog(Dialog *frm)
{
	if (!frm)
		return;

	if (mCurrentDialog == frm)
		SetActiveDialog( NULL );
	
	mDialogs.remove(frm);
	delete frm;
}

void XcsGUIView::SetActiveDialog(Dialog *frm)
{
	mSelectedItems.clear();
	mCurrentDialog = frm;
	RedrawAll();
}

Dialog *XcsGUIView::GetActiveDialog()
{
	return mCurrentDialog;
}

void XcsGUIView::DeleteDialogs()
{
	mSelectedItems.clear();
	int ct = mDialogs.count();
	for (int i=0;i<ct;i++)
	{
		mDialogs[i]->DeleteChildren();
		delete mDialogs[i];
	}
}

int XcsGUIView::GetDialogCount()
{
	return mDialogs.count();
}

Dialog *XcsGUIView::GetDialog(int i)
{
	return mDialogs[i];
}

void XcsGUIView::ClearSelections()
{
	mSelectedItems.clear();
	mPropEdit->SetObject(NULL);
	RedrawAll();
}

bool XcsGUIView::AreObjectsSelected()
{
	return mSelectedItems.count() > 0;
}

int XcsGUIView::GetSelectedCount()
{
	return mSelectedItems.count();
}

Object *XcsGUIView::GetSelected(int i)
{
	return mSelectedItems[i];
}


void XcsGUIView::AlignTopEdges()
{
	if (mSelectedItems.count() > 0)
	{
		Object *first = mSelectedItems[0];
		int ypos = first->GetY();
		for (int i=1;i<mSelectedItems.count();i++)
		{
			wxRect rct = mSelectedItems[i]->GetGeom();
			rct.y = ypos;
			mSelectedItems[i]->SetGeom( rct );
			bDirty = true;
			mFrame->UpdateControlStatus();
		}
		RedrawAll();
	}
}

void XcsGUIView::AlignLeftEdges()
{
	if (mSelectedItems.count() > 0)
	{
		Object *first = mSelectedItems[0];
		int xpos = first->GetX();
		for (int i=1;i<mSelectedItems.count();i++)
		{
			wxRect rct = mSelectedItems[i]->GetGeom();
			rct.x = xpos;
			mSelectedItems[i]->SetGeom( rct );
			bDirty = true;
			mFrame->UpdateControlStatus();
		}
		RedrawAll();
	}
}

void XcsGUIView::AlignRightEdges()
{
	if (mSelectedItems.count() > 0)
	{
		Object *first = mSelectedItems[0];
		wxRect rct = first->GetGeom();
		int xpos = rct.x+rct.width;
		for (int i=1;i<mSelectedItems.count();i++)
		{
			rct = mSelectedItems[i]->GetGeom();
			rct.x = xpos-rct.width;
			mSelectedItems[i]->SetGeom( rct );
			bDirty = true;
			mFrame->UpdateControlStatus();
		}
		RedrawAll();
	}
}

void XcsGUIView::AlignBottomEdges()
{
	if (mSelectedItems.count() > 0)
	{
		Object *first = mSelectedItems[0];
		wxRect rct = first->GetGeom();
		int ypos = rct.y+rct.height;
		for (int i=1;i<mSelectedItems.count();i++)
		{
			wxRect rct = mSelectedItems[i]->GetGeom();
			rct.y = ypos-rct.height;
			mSelectedItems[i]->SetGeom( rct );
			bDirty = true;
			mFrame->UpdateControlStatus();
		}
		RedrawAll();
	}
}

bool XcsGUIView::CanUndo()
{
	return mUndoStack.count() > 0;
}

bool XcsGUIView::CanRedo()
{
	return mRedoStack.count() > 0;
}

wxString XcsGUIView::GetUndoString()
{
	return "<no description>";
}

wxString XcsGUIView::GetRedoString()
{
	return "<no description>";
}

void XcsGUIView::FreeUndoStack()
{
	int ct = mUndoStack.count();
	for (int i=0;i<ct;i++)
		delete mUndoStack[i];

	mUndoStack.clear();
}

void XcsGUIView::FreeRedoStack()
{
	int ct = mRedoStack.count();
	for (int i=0;i<ct;i++)
		delete mRedoStack[i];

	mRedoStack.clear();
}

void XcsGUIView::Undo()
{
	mFrame->UpdateControlStatus();
}

void XcsGUIView::Redo()
{
	mFrame->UpdateControlStatus();
}

void XcsGUIView::ClearAll()
{
	mSelectedItems.clear();
	mPropEdit->SetObject(NULL);
	SetActiveDialog( NULL );
	DeleteDialogs();
	mDialogs.clear();
	mFrame->GetDlgSelector()->UpdateList();
	bDirty = false;
	mFrame->UpdateControlStatus();
}


void XcsGUIView::ShowLoadLog()
{
	wxString ret = wxGetSingleChoice("LoadLog:", "Information", mLoadLog, this, -1, -1, true, 600, 400);
	ret = "";
}

void XcsGUIView::WriteObject(FILE *fp, Object *obj, bool isdialog)
{	
	fprintf(fp, "start %s type %s\n", obj->GetName().c_str(), obj->GetControlInfo()->name);
	wxRect rct = obj->GetGeom();
	fprintf(fp, "geom %d %d %d %d\n", rct.x, rct.y, rct.width, rct.height);
	fprintf(fp, "properties %d\n", obj->NumProps());
	
	int i, count = obj->NumProps();
	for (i=0;i<count;i++)
	{
		obj->GetProp(i)->Write(fp);
	}
	
	
	if (isdialog)
	{
		count = 0;
		Object **children = ((Dialog*)obj)->GetChildren(count);

		fprintf(fp, "beginchildren %d {\n", count);
		for (int i=0;i<count;i++)
			WriteObject(fp, (Object*)children[i], false);
			
		fprintf(fp, "}endchildren\n");
	}
	
	fprintf(fp, "end %s\n", obj->GetName().c_str());
}

bool XcsGUIView::ReadObject(FILE *fp, Dialog *parent, bool isdialog)
{
	wxRect geom;
	char namebuf[256];
	char typebuf[128];
	int nprops;
	
	if (!isdialog && !parent)
	{
		mLoadLog.Add("Database Error(readobject): cannot load a non-dialog object with no parent!");
		wxMessageBox("Database Error(readobject): cannot load a non-dialog object with no parent!\n", "Fatal Error", wxOK | wxICON_ERROR);
		return false;
	}
	
	fscanf(fp, "start %s type %s\n", namebuf, typebuf);
	fscanf(fp, "geom %d %d %d %d\n", &geom.x, &geom.y, &geom.width, &geom.height);
	fscanf(fp, "properties %d\n", &nprops);

	mLoadLog.Add(wxString::Format("Found object '%s'('%s') [%d %d %d %d] (has %d properties).", namebuf, typebuf, geom.x, geom.y, geom.width, geom.height, nprops));
	
	/*dbg("loaddb:readobject: found '%s' [%s] (%d %d %d %d) props:%d\n", namebuf, typebuf,
			geom.x, geom.y, geom.width, geom.height, nprops);*/
	
	Object *obj = NULL;
	Dialog *dlg = NULL;

	if (isdialog != false)
	{
		obj = dlg = MakeNewDialog( wxString(namebuf) );
		mLoadLog.Add("Allocated the dialog ('XcsGUIView::MakeNewDialog')");
	}
	else if (parent != NULL)
	{
		ControlInfo *kls = XcsFindControlInfo( typebuf );
		if (kls)
		{
			obj = new Object( kls, wxString(namebuf) );
			obj->SetDialog( parent );
			mLoadLog.Add("Allocated the object instance");
		}
		else
			return false;
		
		//dbg("loaddb:readobject:: created the object type [%s], setting geom, loading props..\n", typebuf);
	}
	
	if (!obj)
		return false;
	
	if (isdialog)
	{
		wxRect dlgrct = obj->GetGeom();

		geom.x = dlgrct.x;
		geom.y = dlgrct.y;
	}

	if (bDedCompat && !isdialog && parent != NULL)
	{
		wxRect dlgrct = parent->GetGeom();
		geom.x -= 200;
		geom.y -= 120;
		geom.x += dlgrct.x;
		geom.y += dlgrct.y;
	}

	obj->SetGeom( geom );
	
	mLoadLog.Add(wxString::Format("Now loading the %d properties", nprops));

	for (int i=0;i<nprops;i++)
	{
		Property mprop;
		if (!mprop.Read(fp))
			mLoadLog.Add(wxString::Format("Error loading property #%d", i));
		
		mLoadLog.Add(wxString::Format("Loaded property(%d) type: %d '%s'", i, mprop.type, mprop.name.c_str()));
		Property *op = obj->FindProp(mprop.name);
		if (op)
			*op = mprop;
	}
	
	if (isdialog)
	{
		if (dlg == NULL)
		{
			wxMessageBox("Error (readobject):Loading dialog children cannot happen to a null dialog!");
			return false;
		}

		int count = 0;
		fscanf(fp, "beginchildren %d {\n", &count);
		
		mLoadLog.Add(wxString::Format("Now the loading the %d child objects of the dialog.", count));
		for (int i=0;i<count;i++)
			ReadObject(fp, dlg, false);
			
		fscanf(fp, "}endchildren\n");
	}
		
	fscanf(fp, "end %s\n", namebuf);
	mLoadLog.Add(wxString::Format("Finished loading object '%s'.", namebuf)); 
	return true;	
}

bool XcsGUIView::WriteDatabase(FILE *fp)
{
	if (!fp)
		return false;
	
	int ct = mDialogs.count();
	fprintf(fp, "database %d dialogs\n", ct);
	for (int i=0;i<ct;i++)
	{	
		WriteObject(fp, mDialogs[i], true);
	}

	bDirty = false;
	mFrame->UpdateControlStatus();

	return true;
}

bool XcsGUIView::ReadDatabase(FILE *fp, bool dedmode)
{
	if (!fp)
		return false;
	
	bDedCompat = dedmode;

	ClearAll();
	mLoadLog.Clear();

	mLoadLog.Add("ReadDatabase initiated.");

	int numdlgs = 0;
	fscanf(fp, "database %d dialogs\n", &numdlgs);
	mLoadLog.Add(wxString::Format("The database has %d dialogs.", numdlgs));
	for (int i=0;i<numdlgs;i++)
	{
		mLoadLog.Add(wxString::Format("Reading dialog #%d...", i));
		ReadObject(fp, NULL, true);
	}
	
	mFrame->GetDlgSelector()->UpdateList();
	SetActiveDialog( NULL );
	mPropEdit->SetObject( NULL );
	mSelectedItems.clear();

	bDirty = false;
	mFrame->UpdateControlStatus();

	//ShowLoadLog();

	return true;
}

void XcsGUIView::RedrawAll()
{
	wxMemoryDC dc;
	dc.SelectObject(*mBufBitmap);
	
	Painter gfx(&dc);
	gfx.SetColour( GetBackgroundColour() );
	gfx.DrawRectangle(0, 0, GetSize().GetWidth(), GetSize().GetHeight(), true);
	Draw(dc);

	wxClientDC cdc(this);
	cdc.Blit(0, 0, mBufBitmap->GetWidth(), mBufBitmap->GetHeight(), &dc, 0, 0);

	dc.SelectObject( wxNullBitmap );
}

wxRect XcsGUIView::GetFrameRect()
{
	if (mCurrentDialog)
	{
		wxRect rct = mCurrentDialog->GetGeom();
		return wxRect(rct.x-FRAME_WIDTH, rct.y-FRAME_HEIGHT, 
				rct.width+FRAME_WIDTH*2, rct.height+FRAME_HEIGHT+FRAME_WIDTH);
	}
	else
	{
		return wxRect(-1, -1, 0, 0);
	}
}

void XcsGUIView::DrawFrameDecor(Painter &gfx)
{
	wxRect framerct = GetFrameRect();
	gfx.GetDC()->SetClippingRegion( framerct.x, framerct.y, framerct.width, FRAME_HEIGHT );
	
	gfx.SetColour( mFrameColour );
	gfx.DrawRectangle(framerct.x+TITLE_XOFF, framerct.y+FRAME_HEIGHT/2-gfx.GetCharHeight()/2, framerct.width-2*TITLE_XOFF,gfx.GetCharHeight(), true);
	gfx.SetColour( *wxWHITE );
	gfx.SetHelveticaSmallBold();
	gfx.DrawString( framerct.x+TITLE_XOFF, framerct.y+FRAME_HEIGHT/2-gfx.GetCharHeight()/2, mCurrentDialog->GetTitle());
	
	//gfx.SetSysControlColour();
	//gfx.DrawXButton(framerct.x+framerct.width-20, framerct.y+6, 15, 15);
	
	gfx.GetDC()->DestroyClippingRegion();
	
}

void XcsGUIView::Draw(wxDC &dc)
{
	Painter gfx(&dc);

	if (!mCurrentDialog)
	{
		dc.SetFont( wxFont(16, wxROMAN, wxNORMAL, wxBOLD) );
		dc.SetTextForeground( wxColour("grey") );

		dc.DrawText(mDialogs.count() == 0 ? "Create a new dialog to begin." : "No dialog selected.", 20, 20);
		dc.SetFont( wxNullFont );
	}
	else
	{
		wxRect framerct = GetFrameRect();

		// paint the dialog decorations 
		gfx.SetColour( mFrameColour );
		gfx.DrawRaisedPanel(framerct.x, framerct.y, framerct.width, framerct.height);
		DrawFrameDecor( gfx );
	
		// paint the Dialog
		wxRect rct = mCurrentDialog->GetGeom();
		(*mCurrentDialog->GetControlInfo()->paint)(gfx, rct, mCurrentDialog);

		// paint the children
		int count = 0;
		int i;
		Object **objs = mCurrentDialog->GetChildren(count);
		for (i=count-1;i>=0;i--)
		{
			rct = objs[i]->GetGeom();
			gfx.SetDefaults();
			dc.SetClippingRegion( mCurrentDialog->GetGeom() );
			dc.SetClippingRegion(rct);
			
			if (objs[i]->GetControlInfo()->draw_dotted_box)
			{
				gfx.SetLineAttributes(1, Painter::LINE_ON_OFF_DASH, Painter::CAP_BUTT, Painter::JOIN_MITER);
				gfx.DrawRectangle(rct.x, rct.y, rct.width, rct.height, false);
				gfx.SetLineAttributes(1, Painter::LINE_SOLID, Painter::CAP_BUTT, Painter::JOIN_MITER);
			}

			(*objs[i]->GetControlInfo()->paint)(gfx, rct, objs[i]);
			dc.DestroyClippingRegion();
		}

		// paint any selection handles
		gfx.SetColour( mSelectColour );
		count = mSelectedItems.count();
		for (i=0;i<count;i++)
		{
			wxRect rct = mSelectedItems[i]->GetGeom();
			
			if (mSelectedItems[i] != mCurrentDialog)
			{
				// left side
				gfx.DrawRectangle(rct.x - RSZBOXW, rct.y - RSZBOXW, RSZBOXW, RSZBOXW, true);
				gfx.DrawRectangle(rct.x - RSZBOXW, rct.y + rct.height/2 - RSZBOXW/2, RSZBOXW, RSZBOXW, true);
				gfx.DrawRectangle(rct.x - RSZBOXW, rct.y + rct.height, RSZBOXW, RSZBOXW, true);

				// right side
				gfx.DrawRectangle(rct.x + rct.width, rct.y - RSZBOXW, RSZBOXW, RSZBOXW, true);
				gfx.DrawRectangle(rct.x + rct.width, rct.y + rct.height/2 - RSZBOXW/2, RSZBOXW, RSZBOXW, true);
				gfx.DrawRectangle(rct.x + rct.width, rct.y + rct.height, RSZBOXW, RSZBOXW, true);
				
				// bottom
				gfx.DrawRectangle(rct.x + rct.width/2 - RSZBOXW/2, rct.y + rct.height, RSZBOXW, RSZBOXW, true);

				// top
				gfx.DrawRectangle(rct.x + rct.width/2 - RSZBOXW/2, rct.y - RSZBOXW, RSZBOXW, RSZBOXW, true);
			}
			else
			{
				rct = framerct;
				gfx.DrawRectangle(rct.x, rct.y, rct.width, rct.height, false);

				// right
				gfx.DrawRectangle(rct.x + rct.width, rct.y + rct.height/2 - RSZBOXW/2, RSZBOXW, RSZBOXW, true);
				gfx.DrawRectangle(rct.x + rct.width, rct.y + rct.height, RSZBOXW, RSZBOXW, true);
				
				// bottom
				gfx.DrawRectangle(rct.x + rct.width/2 - RSZBOXW/2, rct.y + rct.height, RSZBOXW, RSZBOXW, true);

			}
				
		}
	}
}

int XcsGUIView::IsOverResizeBox(int x, int y, Object *obj)
{
	wxRect rct = obj->GetGeom();

	if (obj == mCurrentDialog)
	{
		rct = GetFrameRect();
	}
	
	if (x >= rct.x-RSZBOXW && x <= rct.x &&
			y >= rct.y-RSZBOXW && y <= rct.y)
		return BOX_TOPLEFT;	
	
	if (x >= rct.x-RSZBOXW && x <= rct.x &&
			y >= rct.y + rct.height/2 - RSZBOXW/2 && y <= rct.y + rct.height/2 + RSZBOXW/2)
		return BOX_LEFT;

	if (x >= rct.x-RSZBOXW && x <= rct.x &&
			y >= rct.y + rct.height && y <= rct.y + rct.height + RSZBOXW)
		return BOX_BOTTOMLEFT;

	
	if (x >= rct.x+rct.width/2-RSZBOXW/2 && x <= rct.x+rct.width/2+RSZBOXW/2 &&
			y >= rct.y-RSZBOXW && y <= rct.y)
		return BOX_TOP;
	
	if (x >= rct.x+rct.width/2-RSZBOXW/2 && x <= rct.x+rct.width/2+RSZBOXW/2 &&
			y >= rct.y + rct.height && y <= rct.y + rct.height + RSZBOXW)
		return BOX_BOTTOM;


	if (x >= rct.x+rct.width && x <= rct.x+rct.width+RSZBOXW &&
			y >= rct.y-RSZBOXW && y <= rct.y)
		return BOX_TOPRIGHT;	
	
	if (x >= rct.x+rct.width && x <= rct.x+rct.width+RSZBOXW &&
			y >= rct.y + rct.height/2 - RSZBOXW/2 && y <= rct.y + rct.height/2 + RSZBOXW/2)
		return BOX_RIGHT;

	if (x >= rct.x+rct.width && x <= rct.x+rct.width+RSZBOXW &&
			y >= rct.y + rct.height && y <= rct.y + rct.height + RSZBOXW)
		return BOX_BOTTOMRIGHT;

	return BOX_NONE;
}

void XcsGUIView::OnLeftButtonDown(wxMouseEvent &evt)
{
	int mx = evt.GetX();
	int my = evt.GetY();

	mOrigX = mx;
	mOrigY = my;

	ClientToScreen(&mOrigX, &mOrigY);
	
	if (mSelectedItems.count() == 1 && (mResizeBox = IsOverResizeBox(mx, my, mSelectedItems[0])))
	{
		if (mSelectedItems[0] == mCurrentDialog &&
			(	mResizeBox == BOX_TOPLEFT ||
				mResizeBox == BOX_TOPRIGHT ||
				mResizeBox == BOX_TOP ||
				mResizeBox == BOX_LEFT ||
				mResizeBox == BOX_BOTTOMLEFT) )
		{
			return;
		}
		else
		{
			// start a resize
			mDiffX = 0;
			mDiffY = 0;
			mDiffW = 0;
			mDiffH = 0;
			bResizeMode = true;
			bResizeModeErase = false;
		}
	}
	else
	{
		// handle a selection procedure	
		Object *select_obj = NULL;
		Object *move_obj = NULL;
		if (mCurrentDialog != NULL)
		{
			wxRect rct;
			int count = 0;
			Object **objs = mCurrentDialog->GetChildren(count);
			for (int i=0;i<count;i++)
			{
				if ( (*objs[i]->GetControlInfo()->iswithin)(mx, my, objs[i]))
				{
					select_obj = objs[i];
					move_obj = objs[i];
					mCurrentDialog->RaiseChild( select_obj );
					break;
				}
			}
		
			if (select_obj == NULL && GetFrameRect().Inside(mx, my))
				select_obj = mCurrentDialog;
		}

		bool redraw = false;

		if (select_obj != NULL && mSelectedItems.find(select_obj) < 0)
		{
			if (!evt.ControlDown())
				mSelectedItems.clear();

			mSelectedItems.append(select_obj);
			redraw = true;
		}
		else if (evt.ControlDown() && select_obj != NULL && mSelectedItems.find(select_obj) >= 0)
		{
			mSelectedItems.remove(select_obj);
			move_obj = NULL;
			redraw = true;
		}
		else if (select_obj == NULL)
		{
			if (mSelectedItems.count() > 0)
				redraw = true;
			mSelectedItems.clear();
		}


		if (select_obj)
		{
			bMoveMode = true;
			bMoveModeErase = false;
			mDiffX = 0;
			mDiffY = 0;
			mDiffW = 0;
			mDiffH = 0;
		}
;

		if (redraw)
			RedrawAll();
		
		if (mPropEdit)
			mPropEdit->SetObject( GetSelectedCount() == 1 ? select_obj : NULL); 
	}
}

void XcsGUIView::OnLeftButtonUp(wxMouseEvent &evt)
{
	if (bMoveMode)
	{
		int i, count;
		count = mSelectedItems.count();
		for (i=0;i<count;i++)
		{
			wxRect rct = mSelectedItems[i]->GetGeom();
			rct.x += mDiffX;
			rct.y += mDiffY;
			XcsSnapPoint(&rct.x, &rct.y);
			mSelectedItems[i]->SetGeom(rct);
			bDirty = true;
			mFrame->UpdateControlStatus();
			mPropEdit->UpdateView();
		}

		bMoveMode = false;
		bMoveModeErase = false;

		RedrawAll();
	}
	else if (bResizeMode && mSelectedItems.count() == 1)
	{	
		wxRect rct = mSelectedItems[0]->GetGeom();
		rct.x += mDiffX;
		rct.y += mDiffY;
		rct.width  += mDiffW;
		rct.height += mDiffH;

		if (rct.width < 5)
			rct.width = 5;
		if (rct.height < 5)
			rct.height = 5;

		mSelectedItems[0]->SetGeom(rct);
		bDirty = true;
		mFrame->UpdateControlStatus();
		mPropEdit->UpdateView();

		bResizeMode = false;
		bResizeModeErase = false;
		RedrawAll();
	}
	
	SetCursor(mStandardCursor);
}

void XcsGUIView::OnRightButtonDown(wxMouseEvent &evt)
{
}

void XcsGUIView::OnKeyEvent(wxKeyEvent &evt)
{
	int dX = 0;
	int dY = 0;

	switch (evt.GetKeyCode())
	{
	case WXK_UP:
		dY = - app_snap_spacing;
		break;
	case WXK_DOWN:
		dY = app_snap_spacing;
		break;
	case WXK_LEFT:
		dX = - app_snap_spacing;
		break;
	case WXK_RIGHT:
		dX = app_snap_spacing;
		break;
	default:
		break;
	}

	if (dX != 0 || dY != 0)
	{
		if (evt.ControlDown())
		{
			dX *= 2;
			dY *= 2;
		}

		int i, count;
		count = mSelectedItems.count();
		for (i=0;i<count;i++)
		{
			if (mSelectedItems[i]->GetDialog() != NULL)
			{
				wxRect rct = mSelectedItems[i]->GetGeom();
				rct.x += dX;
				rct.y += dY;
				XcsSnapPoint(&rct.x, &rct.y);
				mSelectedItems[i]->SetGeom(rct);
				bDirty = true;
				mFrame->UpdateControlStatus();
				mPropEdit->UpdateView();
			}
		}
	
		RedrawAll();
	}
	
}

void XcsGUIView::DrawMoveResizeOutlines()
{
	wxClientDC dc(this);
	dc.SetLogicalFunction( wxINVERT );
	wxBrush brush( *wxWHITE, wxTRANSPARENT );
	wxPen pen(*wxBLACK, 2, wxSOLID);
	pen.SetCap(wxCAP_BUTT);
	pen.SetJoin(wxJOIN_MITER);
	dc.SetBrush(brush);
	dc.SetPen(pen);

	int i, count;
	count = mSelectedItems.count();
	for (i=0;i<count;i++)
	{
		wxRect rct = mSelectedItems[i]->GetGeom();
		if (mSelectedItems[i] == mCurrentDialog)
			rct = GetFrameRect();

		rct.x = rct.x + mDiffX;
		rct.y = rct.y + mDiffY;
		rct.width = rct.width + mDiffW;
		rct.height = rct.height + mDiffH;
		XcsSnapPoint(&rct.x, &rct.y);

		dc.DrawRectangle(rct);
	}

}

void XcsGUIView::SetResizeCursor(int pos)
{
	if (pos < 0)
		pos = mResizeBox;

	if (mSelectedItems.count() == 1 &&
		mSelectedItems[0] == mCurrentDialog &&
		(	pos == BOX_TOPLEFT ||
			pos == BOX_TOPRIGHT ||
			pos == BOX_TOP ||
			pos == BOX_LEFT ||
			pos == BOX_BOTTOMLEFT) )
	{
			return;
	}


	switch(pos)
	{
	case BOX_TOPLEFT:
	case BOX_BOTTOMRIGHT:
		SetCursor( mNwSeCursor );
		break;
	case BOX_TOPRIGHT:
	case BOX_BOTTOMLEFT:
		SetCursor( mNeSwCursor );
		break;
	case BOX_TOP:
	case BOX_BOTTOM:
		SetCursor( mNSCursor );
		break;
	case BOX_LEFT:
	case BOX_RIGHT:
		SetCursor( mWECursor );
		break;
	default:
		SetCursor( mMoveResizeCursor );
	}
}

void XcsGUIView::OnMouseMove(wxMouseEvent &evt)
{
	int mx = evt.GetX();
	int my = evt.GetY();

	int xroot = mx;
	int yroot = my;
	ClientToScreen(&xroot, &yroot);

	if (bMoveMode)
	{
		// can't move the Dialog so remove it from the selected items
		if (mSelectedItems.find( mCurrentDialog ) >= 0)
		{
			mSelectedItems.remove( mCurrentDialog );
			RedrawAll();
		}

		if (bMoveModeErase)
			DrawMoveResizeOutlines();

		mDiffX = xroot - mOrigX;
		mDiffY = yroot - mOrigY;

		XcsSnapPoint(&mDiffX, &mDiffY);

		DrawMoveResizeOutlines();
		bMoveModeErase = true;
	}
	else if (bResizeMode)
	{
		SetResizeCursor();

		if (bResizeModeErase)
			DrawMoveResizeOutlines();

		int diffx = xroot - mOrigX;
		int diffy = yroot - mOrigY;

		switch(mResizeBox)
		{
		case BOX_TOPLEFT:
			mDiffX = diffx;
			mDiffY = diffy;
			mDiffW = -diffx;
			mDiffH = -diffy;
			break;
		case BOX_TOPRIGHT:
			mDiffX = 0;
			mDiffY = diffy;
			mDiffW = diffx;
			mDiffH = -diffy;
			break;
		case BOX_BOTTOMLEFT:
			mDiffX = diffx;
			mDiffY = 0;
			mDiffW = -diffx;
			mDiffH = diffy;
			break;
		case BOX_BOTTOMRIGHT:
			mDiffX = 0;
			mDiffY = 0;
			mDiffW = diffx;
			mDiffH = diffy;
			break;
		case BOX_TOP:
			mDiffX = 0;
			mDiffY = diffy;
			mDiffW = 0;
			mDiffH = -diffy;				
			break;
		case BOX_LEFT:
			mDiffX = diffx;
			mDiffY = 0;
			mDiffW = -diffx;
			mDiffH = 0;
			break;
		case BOX_RIGHT:
			mDiffX = 0;
			mDiffY = 0;
			mDiffW = diffx;
			mDiffH = 0;
			break;
		case BOX_BOTTOM:
			mDiffX = 0;
			mDiffY = 0;
			mDiffW = 0;
			mDiffH = diffy;
			break;
		default:
			break;
		}

		XcsSnapPoint(&mDiffX, &mDiffY);
		XcsSnapPoint(&mDiffW, &mDiffH);

		DrawMoveResizeOutlines();
		bResizeModeErase = true;
	}
	else if (mSelectedItems.count() == 1)
	{
		int box = IsOverResizeBox(mx, my, mSelectedItems[0]);
		if (box)
			SetResizeCursor(box);
		else
			SetCursor( mStandardCursor );
	}
}

void XcsGUIView::OnSize(wxSizeEvent &evt)
{
	wxSize wsz = evt.GetSize();
	if (mBufBitmap->GetWidth() < wsz.GetWidth() ||
		mBufBitmap->GetHeight() < wsz.GetHeight() )
	{
		delete mBufBitmap;
		mBufBitmap = new wxBitmap(wsz.GetWidth(), wsz.GetHeight());
		RedrawAll();
	}
}

void XcsGUIView::OnPaint(wxPaintEvent &evt)
{
	wxPaintDC dc(this);
	PrepareDC(dc);

	dc.BeginDrawing();
	Draw( dc );
	dc.EndDrawing();
}