#include <stdio.h>
#include <stdlib.h>
#include <wx/dc.h>
#include <wx/colour.h>
#include <wx/settings.h>

#include "painter.h"

Painter::Painter(wxDC *mdc)
{
	SetDC(mdc);
}

Painter::~Painter()
{
	/* nothing to do */
}

wxDC *Painter::GetDC()
{
	return mDC;
}

void Painter::SetDC(wxDC *mdc)
{
	this->mDC = mdc;
	SetDefaults();
}

void Painter::SetDefaults()
{
	SetColour(wxColour("black"));
	SetBackgroundColour(wxColour("white"));
	SetHelveticaSmall();
	SetDrawMode(PIXEL_COPY);
	SetLineAttributes(1, LINE_SOLID, CAP_BUTT, JOIN_MITER);	
}

void Painter::SetColour(const wxColour &colour)
{
	mForeground = colour;
	
	mPen.SetColour(colour);
	mBrush.SetColour(colour);
	mDC->SetTextForeground(colour);
}

void Painter::SetSysWindowColour()
{
	SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOW));
}

void Painter::SetSysControlColour()
{
	SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE));
}

void Painter::SetSysHighlightColour()
{
	SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_HIGHLIGHT));
}

void Painter::SetSysTextColour()
{
	SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_BTNTEXT));
}

wxColour Painter::GetColour()
{
	return mForeground;
}

void Painter::SetBackgroundColour(const wxColour &colour)
{
	mBackground = colour;
	mDC->SetTextBackground(colour);	
}

wxColour Painter::GetBackgroundColour()
{
	return mBackground;
}

void Painter::SetFont(const wxFont &fnt)
{
	mFont = fnt;
	mDC->SetFont(fnt);
}

wxFont &Painter::GetFont()
{
	return mFont;
}


void Painter::SetHelveticaTinyBold()
{
	SetFont( wxFont(9, wxSWISS, wxNORMAL, wxBOLD) ); 
}

void Painter::SetHelveticaSmall()
{
	SetFont( wxFont(10, wxSWISS, wxNORMAL, wxNORMAL) ); 
}

void Painter::SetHelveticaSmallBold()
{
	SetFont( wxFont(10, wxSWISS, wxNORMAL, wxBOLD) ); 
}

void Painter::SetHelveticaMedium()
{
	SetFont( wxFont(12, wxSWISS, wxNORMAL, wxNORMAL) ); 
}

void Painter::SetCourierSmall()
{
	SetFont( wxFont(8, wxMODERN, wxNORMAL, wxNORMAL) ); 
}

void Painter::SetCourierMedium()
{
	SetFont( wxFont(10, wxMODERN, wxNORMAL, wxNORMAL) ); 
}

void Painter::SetDrawMode(DrawMode mode)
{
	mDrawMode = mode;
	switch(mode)
	{
	case PIXEL_COPY:
		mDC->SetLogicalFunction(wxCOPY);
		break;
	case PIXEL_INVERT:
		mDC->SetLogicalFunction(wxINVERT);
		break;
	case PIXEL_XOR:
		mDC->SetLogicalFunction(wxXOR);
		break;
	}
}

void Painter::SetLineWidth(unsigned int width)
{
	mLineWidth = width;
	mPen.SetWidth(mLineWidth);
}

void Painter::SetLineAttributes(unsigned int width, PainterStyle linestyle, PainterStyle capstyle, PainterStyle joinstyle)
{
	this->mLineWidth = width;
	this->mLineStyle = linestyle;
	this->mCapStyle = capstyle;
	this->mJoinStyle = joinstyle;
	
	mPen.SetWidth(mLineWidth);
	
	int wxstyle;
	
	wxstyle = wxSOLID;
	switch(mLineStyle)
	{
	case LINE_SOLID:
		wxstyle = wxSOLID;
		break;
	case LINE_ON_OFF_DASH:
		wxstyle = wxSHORT_DASH;
		break;
	case LINE_DOUBLE_DASH:
		wxstyle = wxLONG_DASH;
		break;
	default:
		break;
	}
	mPen.SetStyle(wxstyle);
	
	wxstyle = wxCAP_BUTT;
	switch(mCapStyle)
	{
	case CAP_BUTT:
		wxstyle = wxCAP_BUTT;
		break;
	case CAP_ROUND:
		wxstyle = wxCAP_ROUND;
		break;
	case CAP_PROJECTING:
		wxstyle = wxCAP_PROJECTING;
		break;
	default:
		break;
	}
	mPen.SetCap(wxstyle);
	
	wxstyle = wxJOIN_MITER;
	switch(mJoinStyle)
	{
	case JOIN_MITER:
		wxstyle = wxJOIN_MITER;
		break;
	case JOIN_ROUND:
		wxstyle = wxJOIN_ROUND;
		break;
	case JOIN_BEVEL:
		wxstyle = wxJOIN_BEVEL;
		break;
	default:
		break;
	}
	mPen.SetJoin(wxstyle);
}

int Painter::GetTextWidth(const wxString &text, int length)
{
	int w, h;
	if (length == -1)
	{
		mDC->GetTextExtent(text.c_str(), &w, &h);
	}
	else
	{
		wxString str(text.c_str());
		mDC->GetTextExtent(str.Truncate(length), &w, &h);
	}
	return w;
}

int Painter::GetTextHeight(const wxString &text, int length)
{
	int w, h;
	if (length == -1)
	{
		mDC->GetTextExtent(text.c_str(), &w, &h);
	}
	else
	{
		wxString str(text.c_str());
		mDC->GetTextExtent(str.Truncate(length), &w, &h);
	}
	return h;
}

void Painter::GetTextExtent(const wxString &text, int &width, int &height, int length)
{
	int w, h;
	if (length == -1)
	{
		mDC->GetTextExtent(text.c_str(), &w, &h);
	}
	else
	{
		wxString str(text.c_str());
		mDC->GetTextExtent(str.Truncate(length), &w, &h);
	}
	width = w;
	height = h;
}

int Painter::GetCharHeight()
{
	return mDC->GetCharHeight();
}

void Painter::DrawPoint(int x, int y)
{
	mDC->SetPen(mPen);
	mDC->DrawPoint(x, y);
}

void Painter::DrawLine(int x1, int y1, int x2, int y2)
{
	mDC->SetPen(mPen);
	mDC->DrawLine(x1, y1, x2, y2);
}

void Painter::DrawLines(wxPoint *points, int npoints)
{
	mDC->SetPen(mPen);
	mDC->DrawLines(npoints, points);
}

void Painter::DrawPolygon(wxPoint *points, int npoints, bool fill)
{
	mDC->SetPen(mPen);
	
	if (fill)
		mBrush.SetStyle(wxSOLID);
	else
		mBrush.SetStyle(wxTRANSPARENT);
		
	mDC->SetBrush(mBrush);
	mDC->DrawPolygon(npoints, points);
}

void Painter::DrawRectangle(int x, int y, int width, int height, bool fill)
{
	mDC->SetPen(mPen);
	
	if (fill)
		mBrush.SetStyle(wxSOLID);
	else
		mBrush.SetStyle(wxTRANSPARENT);
		
	mDC->SetBrush(mBrush);
	mDC->DrawRectangle(x, y, width, height);
}

void Painter::DrawCheckMark(int x, int y, int width, int height, int thick)
{
	mDC->SetPen(mPen);
	mDC->SetBrush(mBrush);

	int lnws = mLineWidth;
	PainterStyle lstyle = mLineStyle;
	PainterStyle lcap = mCapStyle;
	PainterStyle ljoin = mJoinStyle;

	SetLineAttributes(thick, LINE_SOLID, CAP_ROUND, JOIN_ROUND);

	wxRect geom(x+1, y+1, width-2, height-2);

	int x1 = x + .2*width;
	int x2 = x + .4*width;
	int x3 = x + .8*width;

	int y1 = y + .4*height;
	int y2 = y + .7*height;
	int y3 = y + .3*height;

	DrawLine(x1, y1, x2, y2);
	DrawLine(x2, y2, x3, y3);   
	

	SetLineAttributes(lnws, lstyle, lcap, ljoin);
}

void Painter::DrawRoundedRect(int x, int y, int width, int height, bool fill, float curvature)
{
	mDC->SetPen(mPen);
	
	if (fill)
		mBrush.SetStyle(wxSOLID);
	else
		mBrush.SetStyle(wxTRANSPARENT);
		
	mDC->SetBrush(mBrush);
	mDC->DrawRoundedRectangle(x, y, width, height, curvature);
}

void Painter::DrawEllipse(int x, int y, int width, int height, float start, float end, bool fill)
{
	if (fill)
		mBrush.SetStyle(wxSOLID);
	else
		mBrush.SetStyle(wxTRANSPARENT);

	mDC->SetPen(mPen);
	mDC->SetBrush(mBrush);

	mDC->DrawEllipticArc(x, y, width, height, start, end);
}

void Painter::DrawCircle(int x, int y, int radius, bool fill)
{
	if (fill)
		mBrush.SetStyle(wxSOLID);
	else
		mBrush.SetStyle(wxTRANSPARENT);
	
	mDC->SetPen(mPen);
	mDC->SetBrush(mBrush);
	
	mDC->DrawCircle(x, y, radius);
}

void Painter::DrawImageString(int x, int y, const wxString &string)
{
	DrawImageText(x, y, string.c_str(), string.length());
}

void Painter::DrawImageText(int x, int y, const char *text, int text_length)
{
	int w, h;
	mDC->GetTextExtent(text, &w, &h);
	SetColour(mBackground);
	DrawRectangle(x, y, w, h, true);
	SetColour(mForeground);
	
	mDC->DrawText(wxString(text).Truncate(text_length), x, y);
}

void Painter::DrawString(int x, int y, const wxString &string)
{
	DrawText(x, y, string, string.length());
}

void Painter::DrawText(int x, int y, const char *text, int text_length)
{
	mDC->DrawText(wxString(text).Truncate(text_length), x, y);
}

wxImage *Painter::Scale(wxImage *img, int new_width, int new_height)
{
	unsigned char *src = img->GetData();
	int src_width = img->GetWidth();
	int src_height = img->GetHeight();
	
	int index_new;
	
	double sx = (double)new_width / (double)src_width;
	double sy = (double)new_height / (double)src_height;
	
	wxImage *target = new wxImage();
	target->Create(new_width, new_height);
	
	unsigned char *pixels = target->GetData(); 
	
	double x_old, y_old;
		
	for (int y=0; y < new_height; y++)
	{
		for (int x=0; x < new_width; x++)
		{		
			index_new = (y * new_width) + x;
						
			x_old = (((double)x / sx) > src_width - 2) ? src_width - 2 : (double)x / sx;
			y_old = (((double)y / sy) > src_height - 2) ? src_height - 2 : (double)y / sy;
				
			pixels[index_new * 3 /* r */] = (unsigned char) (
					(src[((int)y_old * src_width + (int)(x_old)) * 3 /* r */] * 
					  ((int)(x_old + 1) - x_old) + 
					    src[((int)y_old * src_width + (int)(x_old + 1)) * 3 /* r */] 
					    * (x_old - (int)x_old)) * ((int)(y_old + 1) - y_old)    
					+ (src[((int)(y_old + 1) * src_width + (int)(x_old)) * 3 /* r */] * 
					    ((int)(x_old + 1) - x_old) + 
					    src[((int)(y_old + 1) * src_width + (int)(x_old + 1)) * 3 /* r */] 
					    * (x_old - (int)x_old)) * (y_old - (int)y_old) );
					    
			pixels[index_new * 3 + 1 /* g */] = (unsigned char) (
					(src[((int)y_old * src_width + (int)(x_old)) * 3 + 1 /* g */] * 
					  ((int)(x_old + 1) - x_old) + 
					    src[((int)y_old * src_width + (int)(x_old + 1)) * 3 + 1 /* g */] 
					    * (x_old - (int)x_old)) * ((int)(y_old + 1) - y_old) 
					+ (src[((int)(y_old + 1) * src_width + (int)(x_old)) * 3 + 1 /* g */] * 
					    ((int)(x_old + 1) - x_old) + 
					    src[((int)(y_old + 1) * src_width + (int)(x_old + 1)) * 3 + 1 /* g */] 
					    * (x_old - (int)x_old)) * (y_old - (int)y_old) );
					    
			pixels[index_new * 3 + 2 /* b */] = (unsigned char) (
					(src[((int)y_old * src_width + (int)(x_old)) * 3 + 2 /* b */] * 
					  ((int)(x_old + 1) - x_old) + 
					    src[((int)y_old * src_width + (int)(x_old + 1)) * 3 + 2 /* b */] 
					    * (x_old - (int)x_old)) * ((int)(y_old + 1) - y_old) 
					+ (src[((int)(y_old + 1) * src_width + (int)(x_old)) * 3 + 2 /* b */] * 
					    ((int)(x_old + 1) - x_old) + 
					    src[((int)(y_old + 1) * src_width + (int)(x_old + 1)) * 3 + 2 /* b */] 
					    * (x_old - (int)x_old)) * (y_old - (int)y_old) );
		}
	}
	
	return target;
}

void Painter::DrawXButton(int x, int y, int width, int height)
{
	DrawRaisedPanel(x, y, width, height);
	
	wxColour saved = GetColour();
	int lnws = mLineWidth;
	PainterStyle lstyle = mLineStyle;
	PainterStyle lcap = mCapStyle;
	PainterStyle ljoin = mJoinStyle;


	SetColour("black");
	SetLineAttributes(2, LINE_SOLID, CAP_ROUND, JOIN_ROUND);

	DrawLine(x+3, y+3, x+width-4, y+height-4);
	DrawLine(x+width-4, y+3, x+3, y+height-4);

	SetColour(saved);
	SetLineAttributes(lnws, lstyle, lcap, ljoin);
}


void Painter::DrawRaisedPanel(int x, int y, int width, int height)
{	
	
	DrawRectangle(x, y, width, height, true);
	
	wxColour saved = GetColour();
	SetColour( *wxWHITE );

	DrawLine(x, 				y+1, 				x+width-1, 		y+1);
	DrawLine(x, 				y+1, 					x, 				y+height-1);
	DrawLine(x+1, 				y+1, 				x+width-2, 		y+1);
	
	SetColour( *wxLIGHT_GREY );
	DrawLine(x+1, 			y+height-2,			x+width-2, 		y+height-2);
	DrawLine(x+width-2, 	y+2, 				x+width-2, 		y+height-2);
	
	SetColour( *wxBLACK );
	DrawLine(x, 				y+height-1, 	x+width-1, 		y+height-1);
	DrawLine(x+width-1, 		y, 				x+width-1, 		y+height);	
	
	
	SetColour(saved);
}

void Painter::DrawSunkenPanel(int x, int y, int width, int height)
{
	DrawRectangle(x, y, width, height, true);

	wxColour saved = GetColour();
	
	SetColour( *wxBLACK );
	DrawRectangle(x, y, width-1, height-1, false);
	
	SetColour( "gray" );

	DrawRectangle(x+1, y+1, width-2, height-2, false);
	SetColour(saved);
}

void Painter::DrawEngravedPanel(int x, int y, int width, int height, bool fill)
{
	wxColour saved = GetColour();
	
	if (fill)
		DrawRectangle(x, y, width, height, true);
	
	SetColour( "gray" );
	DrawRectangle(x, y, width-2, height-2, false);
	
	SetColour( *wxWHITE );
	DrawRectangle(x+1, y+1, width-2, height-2, false);
	SetColour(saved);
}

void Painter::DrawScrollBar(bool vertical, int x, int y, int width, int height)
{
	wxColour old = GetColour();
	
	SetColour("light grey");
	DrawRectangle(x, y, width, height, true);	

	SetColour( old );
	if (vertical)
	{
		DrawArrowButton(UP, x, y, width, width);
		if (height > 2.5*width)
			DrawRaisedPanel(x, y+width+1, width, 0.3*height);

		DrawArrowButton(DOWN, x, y+height-width, width, width);
	}
	else
	{
		DrawArrowButton(LEFT, x, y, height, height);
		if (width > 2.5*height)
			DrawRaisedPanel(x+height+1, y, 0.3*width, height);

		DrawArrowButton(RIGHT, x+width-height, y, height, height);
	}
	SetColour(old);
}

void Painter::DrawArrowButton(ArrowType type, int x, int y, int width, int height)
{
	int asize = width < height ? width/2 : height/2;
	
	wxColour saved = GetColour();
	DrawRaisedPanel(x, y, width, height);
	SetColour( *wxBLACK );
	
	switch(type)
	{
	case UP:
	case DOWN:
		DrawArrow(type, x+(width-asize)/2, y+(height-asize)/2, asize, asize);
		break;
	default:
		DrawArrow(type, x+(width-asize)/2, y+(height-asize)/2, asize, asize);
	}
	
	SetColour(saved);
}

void Painter::DrawArrow(ArrowType type, int x, int y, int width, int height)
{

	wxPoint pts[3];
	switch(type)
	{
	case RIGHT:
      pts[0] = wxPoint(x,y);
      pts[1] = wxPoint(x, y+height);
      pts[2] = wxPoint(x+width, y+height/2);
      break;
   case LEFT:
      pts[0] = wxPoint(x+width,y);
      pts[1] = wxPoint(x+width, y+height);
      pts[2] = wxPoint(x, y+height/2);
      break;
   case UP:
      pts[0] = wxPoint(x,y+height);
      pts[1] = wxPoint(x+width, y+height);
      pts[2] = wxPoint(x+width/2, y);
      break;
   case DOWN:
      pts[0] = wxPoint(x,y);
      pts[1] = wxPoint(x+width, y);
      pts[2] = wxPoint(x+width/2, y+height);
      break;
   default:
   	return;
  	}

	DrawPolygon(pts, 3, true);
}

void Painter::DrawOutline(int x, int y, int width, int height, bool rev)
{
	wxColour saved = GetColour();
	
	if (rev)
		SetColour( "grey" );
	else
		SetColour( *wxBLACK );
		
	DrawRectangle(x, y, width-2, height-2, false);
	
	if (rev)
		SetColour(*wxBLACK);
	else
		SetColour("grey");
		
	DrawRectangle(x+1, y+1, width-2, height-2, false);
	
	SetColour(saved);
}
