#include <wx/wx.h>
#include <wx/colour.h>
#include <wx/colordlg.h>

#include "editctrls.h"


#define MIN(a,b) a<b?a:b
#define MAX(a,b) a>b?a:b

EditorCtrl::EditorCtrl(wxWindow *hwnd)
	: geom(1, 1, 20, 10)
{
	this->focus = false;
	this->hwnd = hwnd;
}

EditorCtrl::~EditorCtrl()
{
	/* nothing to do */
}

void EditorCtrl::setfocus(bool focus)
{
	this->focus = focus;
	redraw();
}

bool EditorCtrl::hasfocus()
{
	return this->focus;
}

void EditorCtrl::setgeom(const wxRect &rct)
{
	geom = rct;
}

wxRect EditorCtrl::getgeom()
{
	return geom;
}

int EditorCtrl::getx()
{
	return geom.x;
}

int EditorCtrl::gety()
{
	return geom.y;
}

int EditorCtrl::getwidth()
{
	return geom.width;
}

int EditorCtrl::getheight()
{
	return geom.height;
}

void EditorCtrl::redraw()
{
	wxClientDC dc(hwnd);
	dc.SetClippingRegion(geom);
	Painter gfx(&dc);
	draw(gfx);
	dc.DestroyClippingRegion();
}

wxWindow *EditorCtrl::gethwnd()
{
	return this->hwnd;
}


checkedit::checkedit(wxWindow *hwnd)
	: EditorCtrl(hwnd)
{
	togglecb = NULL;
	toggle_data = NULL;
	value = false;
}

void checkedit::togglecallback( void (*cb)(checkedit *e, void *data), void *data)
{
	togglecb = cb;
	toggle_data = data;
}

void checkedit::setvalue(bool b)
{
	bool x = false;
	if (b != value)
		x = true;

	value = b;

	if (x)
		redraw();
}

bool checkedit::getvalue()
{
	return value;
}

void checkedit::draw(Painter &gfx)
{
	wxRect chkrect(geom.x+3, geom.y+4, geom.height-6, geom.height-6);

	gfx.SetColour(*wxWHITE);
	gfx.DrawRectangle(chkrect.x, chkrect.y, chkrect.width, chkrect.height, true);

	gfx.SetColour( *wxBLACK );

	if (value)
		gfx.DrawCheckMark(chkrect.x, chkrect.y, chkrect.width, chkrect.height, 3);

	gfx.DrawRectangle(chkrect.x, chkrect.y, chkrect.width, chkrect.height, false);
}

void checkedit::onchar(wxKeyEvent &evt)
{
	if (evt.GetKeyCode() == ' ')
	{
		value = !value;
		redraw();
		if (togglecb != NULL)
			(*togglecb)(this, toggle_data);
	}
}

void checkedit::onleftdown(wxMouseEvent &evt)
{
	value = !value;
	redraw();
	if (togglecb != NULL)
		(*togglecb)(this, toggle_data);
}

void checkedit::ondblclick(wxMouseEvent &evt)
{
	onleftdown(evt);
}

enum { ID_SE_Ok, ID_SE_Cancel, ID_SE_Add,
	ID_SE_Remove, ID_SE_Clear,
	ID_SE_InsertBefore, ID_SE_InsertAfter };

BEGIN_EVENT_TABLE( StrEditDialog, wxDialog )
	EVT_BUTTON(ID_SE_Ok, wxDialog::OnOK)
	EVT_BUTTON(ID_SE_Cancel, wxDialog::OnCancel)

	EVT_BUTTON(ID_SE_Add, StrEditDialog::OnCommand)
	EVT_BUTTON(ID_SE_Remove, StrEditDialog::OnCommand)
	EVT_BUTTON(ID_SE_InsertBefore, StrEditDialog::OnCommand)
	EVT_BUTTON(ID_SE_InsertAfter, StrEditDialog::OnCommand)
	EVT_BUTTON(ID_SE_Clear, StrEditDialog::OnCommand)
END_EVENT_TABLE()

StrEditDialog::StrEditDialog(wxWindow *parent)
	: wxDialog(parent, -1, "String List Editor",
	    wxDefaultPosition, wxSize(300, 350), wxDIALOG_MODAL | wxCAPTION )
{
	wxButton *btn;

	mListBox = new wxListBox(this, -1, wxPoint(10, 10),	wxSize(170, 230));

	mAdd = new wxButton(this, ID_SE_Add, "Add...", wxPoint(190, 10), wxSize(100, 25));
	mInsertBefore = new wxButton(this, ID_SE_InsertBefore, "Insert Before...", wxPoint(190, 40), wxSize(100, 25));
	mInsertAfter = new wxButton(this, ID_SE_InsertAfter, "Insert After...", wxPoint(190, 70), wxSize(100, 25));
	mRemove = new wxButton(this, ID_SE_Remove, "Remove", wxPoint(190, 100), wxSize(100, 25));
	mClear = new wxButton(this, ID_SE_Clear, "Clear", wxPoint(190, 130), wxSize(100, 25));

	btn = new wxButton(this, ID_SE_Ok, "Ok", wxPoint(190, 185), wxSize(100, 25));
	btn = new wxButton(this, ID_SE_Cancel, "Cancel", wxPoint(190, 215), wxSize(100, 25));

	SetClientSize(300, 250);
}

bool StrEditDialog::RunDialog(wxArrayString &array)
{
	unsigned int i;
	for (i=0;i<array.Count();i++)
	{
		mListBox->Append(array[i]);
	}

	CenterOnScreen();

	if (ShowModal() == wxID_OK)
	{
		array.Clear();
		for (i=0;i<(unsigned int)mListBox->GetCount();i++)
			array.Add(mListBox->GetString(i));

		return true;
	}
	else
	{
		return false;
	}

}

void StrEditDialog::OnCommand(wxCommandEvent &evt)
{
	wxString str;
	int id;

	switch(evt.GetId())
	{
	case ID_SE_Add:
		str = wxGetTextFromUser("Enter string value:", "Add String", "", this);
		if (str != wxEmptyString)
			mListBox->Append(str);
		break;
	case ID_SE_InsertAfter:
	case ID_SE_InsertBefore:
		str = wxGetTextFromUser("Enter string value:", "Add String", "", this);
		if (str != wxEmptyString)
		{
			id = mListBox->GetSelection();
			if (id >= 0 && id < mListBox->GetCount())
			{
				int idx = evt.GetId() == ID_SE_InsertAfter ? id+1 : id;
				mListBox->InsertItems(1, &str, idx);
			}
		}
		break;
	case ID_SE_Remove:
		id = mListBox->GetSelection();
		if (id >= 0 && id < mListBox->GetCount())
			mListBox->Delete(id);
	
		break;
	case ID_SE_Clear:
		mListBox->Clear();
		break;
	default:
		break;
	}
}

strlistedit::strlistedit(wxWindow *hwnd)
	: EditorCtrl(hwnd)
{
	changedcb = NULL;
	changed_data = NULL;
}

void strlistedit::changedcallback( void (*cb)(strlistedit *e, void *data), void *data)
{
	changedcb = cb;
	changed_data = data;
}

void strlistedit::setstrings(const wxArrayString &strs)
{
	value = strs;
}

wxArrayString strlistedit::getstrings()
{
	return value;
}

void strlistedit::draw(Painter &gfx)
{
	int texty = geom.y + geom.height/2-gfx.GetCharHeight()/2;

	gfx.SetHelveticaTinyBold();
	gfx.SetSysControlColour();
	gfx.DrawRaisedPanel(geom.x+3, geom.y+1, geom.height-2, geom.height-2);
	gfx.SetColour(*wxBLACK);
	gfx.DrawString(geom.x+3+ geom.height/2-1-gfx.GetTextWidth("...")/2, texty, "...");
}

void strlistedit::onleftdown(wxMouseEvent &evt)
{
	StrEditDialog dlg(gethwnd());
	if (dlg.RunDialog(value))
	{
		if (changedcb != NULL)
			(*changedcb)(this, changed_data);
	}
}

colouredit::colouredit(wxWindow *hwnd)
	: EditorCtrl(hwnd)
{
	changedcb = NULL;
	changed_data = NULL;
}

void colouredit::changedcallback( void (*cb)(colouredit *e, void *data), void *data)
{
	changedcb = cb;
	changed_data = data;
}

void colouredit::setcolour(const wxColour &c)
{
	value = c;
	redraw();
}

wxColour colouredit::getcolour()
{
	return value;
}

void colouredit::draw(Painter &gfx)
{
	gfx.SetColour(value);
	gfx.DrawRaisedPanel(geom.x+3, geom.y+1, geom.height-2, geom.height-2);
}

void colouredit::onleftdown(wxMouseEvent &evt)
{
	wxColour c = wxGetColourFromUser(gethwnd(), value);
	if (c.Ok())
	{
		value = c;
		redraw();
		if (changedcb != NULL)
			(*changedcb)(this, changed_data);
	}
}



#define TMRSCROLL 1
#define TMRSPINNER 2

entry::entry(wxWindow *hwnd)
	: EditorCtrl(hwnd)
{
	activatecb = NULL;
	activate_data = NULL;
	changedcb = NULL;
	changed_data = NULL;

	buffer = NULL;
	buffer_size = text_length = 0;
	
	growbuffer();
	
	cursor_pos = 0;
	offset = 0;
	selection_anchor = 0;
	selection_drag = 0;
	margin = 4;
	drag_scrolling = false;
	scrolling_left = false;
	mouse_down = false;
	
	spinning = false;
	incrementing = false;
	
	mode = entry::STRING;
	ndigits = 2;
	ivalue = 0;
	fvalue = 0.0;
	minimum = -1000000.0;
	maximum = 1000000.0;
	step = 1.0;
	editable = true;
	timer_id = -1;
	spintimer_id = -1;
	curtimer = 0;
}

entry::~entry()
{
	wxTimer::Stop();
	delete [] buffer;
}


void entry::activatecallback( void (*cb)(entry *e, void *data), void *data)
{
	activatecb = cb;
	activate_data = data;
}

void entry::changedcallback( void (*cb)(entry *e, void *data), void *data)
{
	changedcb = cb;
	changed_data = data;
}

int entry::textlength()
{
	return text_length;
}

void entry::settext(const char *t)
{
	clear();
	cursor_pos = 0;
	ins(cursor_pos, t);
	cursor_pos = 0;
}

const char *entry::gettext()
{
	return buffer;
}

void entry::ins(int &start_position, const char *tins, int length)
{
	int i;

	if (!tins || length==0)
		return;
		
	if (length < 0)
	{
		length = strlen(tins);
		if (length <= 0)
			return;
	}
		
	if (mode == entry::BOOLEAN)
		return;
	
	char *text = strdup(tins);
	
	for (i=0;i<length;i++)
	{
		if (!charok(text[i]))
		{
			text[i] = '\0';
			length = i;
			break;
		}
	}
	
	if (length <= 0)
		return;
			
	
	if (hasselectedtext())
		deletekey();
		
	if (start_position < 0)
		start_position = 0;
	else if (start_position > text_length)
		start_position = text_length;
	
	int end_position = start_position + length;
	int last_position = length + text_length;
		
	while (last_position >= buffer_size)
		growbuffer();
	
	for (i = last_position - 1; i >= end_position; i--)
		buffer[i] = buffer[i - (end_position - start_position)];
	
	for (i = start_position; i < end_position; i++)
		buffer[i] = text[i - start_position];
	
	text_length += length;
	start_position = end_position;
	selection_anchor = start_position;
	selection_drag = start_position;

	buffer[text_length] = '\0';
	
	if (mode == entry::INTEGER)
		ivalue = atoi(buffer);
	else if (mode == entry::FLOAT)
		fvalue = atof(buffer);
		
	redraw();
	
	if (changedcb != NULL)
		(*changedcb)(this, changed_data);
	
	free(text); /* allocated with strdup */
}

void entry::del(int start, int length)
{
	int i;

	if (mode == BOOLEAN)
		return;
	
	if (length < 0 || length > text_length)
		length = text_length - start;
	else if (length == 0)
		return;
		
	if (start < 0)
		start = 0;
	
	for (i = start + length; i < text_length; i++)
		buffer[i - length] = buffer[i];
	
	for (i = text_length - length; i < text_length; i++)
		buffer[i] = '\0';
		
	text_length -= length;
	
	cursor_pos = start;
	selection_anchor = start;
	selection_drag = start;
	
	if (mode == INTEGER)
		ivalue = atoi(buffer);
	else if (mode == FLOAT)
		fvalue = atof(buffer);

	redraw();
	
	if (changedcb != NULL)
		(*changedcb)(this, changed_data);
}

void entry::clear()
{
	del(0, -1);
}

void entry::backspacekey()
{
	if (hasselectedtext())
	{
		del(selstart(),selend() - selstart());
	}
	else if (cursor_pos > 0)
	{
		del(cursor_pos-1, 1);
	}
}

void entry::deletekey()
{
	int tmp_pos = cursor_pos;
	if (hasselectedtext())
	{
		del(selstart(),selend() - selstart());
	}
	else if (cursor_pos < text_length)
	{
		del(cursor_pos, 1);
		cursor_pos = tmp_pos;
	}
}
int entry::cursorpos()
{
	return cursor_pos;
}

void entry::setcursorpos(int pos)
{
	if (pos < 0)
		pos = 0;
	if (pos > text_length)
		pos = text_length;
	
	cursor_pos = pos;
	
	redraw();
}

void entry::cursforward(bool mark)
{
	if (cursor_pos < text_length || (!mark && hasselectedtext()))
	{
		int min_pt = MIN(cursor_pos, selstart());
		cursor_pos++;
		int max_pt = MAX(cursor_pos, selend());

		if (mark)
		{
			selection_drag = cursor_pos;
		}
		else
		{
			selection_anchor = selection_drag = cursor_pos;
		}

		min_pt = max_pt + 1; // BOGUS SO THE COMPILER DOESN'T COMPLAIN
									// USEFUL LATER WHEN OPTIMIZING PAINTING
									// OF THE WIDGET		
		redraw();
	}		
}

void entry::cursbackward(bool mark)
{
	if (cursor_pos > 0 || (!mark && hasselectedtext()))
	{
		cursor_pos--;
		int min_pt = MIN(cursor_pos, selstart());
		int max_pt = MAX(cursor_pos, selend());
		
		if (mark)
		{
			selection_drag = cursor_pos;
		}
		else
		{
			selection_anchor = selection_drag = cursor_pos;
		}
		
		min_pt = max_pt + 1; // BOGUS SO THE COMPILER DOESN'T COMPLAIN
									// USEFUL LATER WHEN OPTIMIZING PAINTING
									// OF THE WIDGET

		redraw();
	}
}

void entry::home(bool mark)
{
	if (cursor_pos != 0 || (!mark && hasselectedtext()))
	{
		cursor_pos = 0;
		if (mark)
		{
			selection_drag = cursor_pos;
		}
		else
		{
			selection_anchor = selection_drag = cursor_pos;
		}
		
		offset = 0;
		redraw();
	}			
}

void entry::end(bool mark)
{
	if (cursor_pos < text_length || (!mark && hasselectedtext()))
	{
		int mo = showlastpartoffset( &buffer[offset], eawidth());
		cursor_pos = text_length;
		if (mark)
		{
			selection_drag = text_length;
		}
		else
		{
			selection_anchor = selection_drag = cursor_pos;
		}
		
		if (mo>0)
			offset+=mo;
		
		redraw();
	}
}

bool entry::hasselectedtext()
{
	return selection_anchor != selection_drag;
}

int entry::sellength()
{
	return selend() - selstart();
}

void entry::selectall()
{
	select(0, text_length);
}

void entry::deselect()
{
	select(cursor_pos, cursor_pos);
}

void entry::select(int start, int end)
{
	if (start < 0)
		start = 0;
	if (end < 0)
		end = 0;
	if (start >= text_length)
		start = text_length;
	if (end >= text_length)
		end = text_length;
		
	selection_anchor = start;
	selection_drag = end;
	redraw();
}

#define EASPINWIDTH 15

void entry::draw(Painter &gfx)
{
	char *disp_text;
	int x, y, width, height;
	x = geom.x;
	y = geom.y;
	width = eawidth();
	height = geom.height;	

	if (offset > cursor_pos)
		offset = cursor_pos;

	gfx.SetHelveticaSmall();
		
	int x3 = gfx.GetTextWidth(buffer+offset, cursor_pos - offset) + 2 * margin;
	
	while (x3 >= width )
	{
		x3 -= gfx.GetTextWidth(buffer+offset, 1);
		offset++;
	}

	gfx.SetColour(*wxWHITE);
	gfx.DrawRectangle(x, y, width, height, true);
			
	disp_text = buffer+offset;
	
	int ypos = 2;
			
	if (strcmp(disp_text, "") != 0)
	{
		int selstart, selend;
		int chars_visible;
		
		chars_visible = lastcharvisible() - offset;
		if ( disp_text[chars_visible] != '\0')
			chars_visible++;
		
		selstart = this->selstart();
		selend = this->selend();
		
		int mark_start, mark_end;
		
		if (selstart > offset)
		{
			if (selstart < offset + chars_visible)
				mark_start = selstart - offset;
			else
				mark_start = chars_visible;
		}
		else
		{
			mark_start = 0;
		}
		
		if (selend > offset )
		{
			if (selend < offset + chars_visible)
				mark_end = selend - offset;
			else
				mark_end = chars_visible;
		}
		else
		{
			mark_end = 0;
		}
		
		if (mark_start != mark_end)
		{
			char *marked = disp_text+mark_start;
			int marked_len = mark_end - mark_start;
			
			int xpos1 = margin + 2 + gfx.GetTextWidth(disp_text, mark_start);
			int xpos2 = xpos1 + gfx.GetTextWidth(marked, marked_len) - 1;
			
			if (hasfocus())
				gfx.SetColour(*wxBLUE);
			else
				gfx.SetColour("light grey");

			gfx.DrawRectangle(x+xpos1, y+margin,
										xpos2 - xpos1, height - margin - margin,
										true);

			gfx.SetColour(*wxWHITE);
			gfx.DrawText(x+xpos1, y+ypos, marked, marked_len);
		}
		
		
		gfx.SetColour(*wxBLACK);

		if (mark_start != 0)
			gfx.DrawText( x+margin + 2, y+ypos, disp_text, mark_start);

		if (mark_end != chars_visible)
		{
			char *rest = disp_text + mark_end;
			int rest_len = chars_visible - mark_end;
			
			gfx.DrawText(x+margin + 2 + gfx.GetTextWidth(disp_text, mark_end), y+ypos, rest, rest_len);
		}
	}
			
			
	
	if (hasfocus())
	{
		gfx.SetColour(*wxBLACK);
		gfx.DrawRectangle(x+1,y+1,width-2,height-2, false);
	}
		
	if ( hasfocus() && mode != BOOLEAN)
	{
		int curxpos = margin + 2;
		curxpos += offset > cursor_pos ? -1 : gfx.GetTextWidth(disp_text, cursor_pos - offset) - 1;
		gfx.SetLineWidth(2);
		gfx.DrawLine(x+curxpos, y+margin+1, x+curxpos, y+height - margin-1);
		gfx.SetLineWidth(1);
	}
	
	if (mode != STRING)
	{
		if (isoutofrange())
			gfx.SetColour(*wxRED);
		else
			gfx.SetColour("white");
		
		gfx.DrawArrowButton(Painter::UP, x+width, y, EASPINWIDTH-1, height/2);
		gfx.DrawArrowButton(Painter::DOWN, x+width, y+height/2, EASPINWIDTH-1, height/2);
	}
}


void entry::ondblclick(wxMouseEvent &evt)
{
	int ret = onspinner(evt.GetX(), evt.GetY());
	if (ret == 0)
	{
		int start, end;
		start = end = cursor_pos;
		if (text_length > 0)
		{
			while (start >= 0 && 
				(isalnum(buffer[start]) || buffer[start] == '_') )
				start--;
			
			while (end < text_length && 
				(isalnum(buffer[end]) || buffer[end] == '_') )
				end++;
			
			select(start+1, end);
		}
	}
	else
	{
		if (ret>0)
			increment();
		else
			decrement();	
	}
}

void entry::onchar(wxKeyEvent &evt)
{
	if (!hasfocus())
		return;

	int key = evt.GetKeyCode();
	switch(key)
	{
	case WXK_BACK:
		backspacekey();
		break;
	case WXK_DELETE:
		deletekey();
		break;
	case WXK_HOME:
		home(evt.ShiftDown());
		break;
	case WXK_END:
		end(evt.ShiftDown());
		break;
	case WXK_LEFT:
		cursbackward(evt.ShiftDown());
		break;
	case WXK_RIGHT:
		cursforward(evt.ShiftDown());
		break;
	case WXK_UP:
		increment();
		break;
	case WXK_DOWN:
		decrement();
		break;			
	case WXK_RETURN:
	  {
		if (activatecb != NULL)
			(*activatecb)(this, activate_data);
	  }
	default:
		if (key >= 32 && key < 127)
		{
			if (key == 'a' && evt.ControlDown())
			{
				selectall();
			}
			else if (editable)
			{
				char s[2];
				s[1] = 0;
				s[0] = key;
				ins(cursor_pos, s, 1);
			}
			
			break;
		}
	}
}
	
void entry::onmousemove(wxMouseEvent &evt)
{
	if (!hasfocus())
		return;


	if (mouse_down)
	{
		int x = evt.GetX() - this->getx();
		
		if (x < margin || x > this->eawidth() - margin)
		{
			scrolling_left = x < margin;
			if (drag_scrolling == false)
			{
				drag_scrolling = true;
				if (scrolling_left)
					selection_drag = offset;
				else
					selection_drag = lastcharvisible();
				
				curtimer = TMRSCROLL;
				wxTimer::Start(110, false);
			}
			else
			{
				if (scrolling_left)
					cursbackward(true);
				else
					cursforward(true);
			}
		}
		else
		{
			drag_scrolling = false;
			cursor_pos = offset + xpostocursorpos(buffer+offset, x - margin);
			selection_drag = cursor_pos;
			redraw();
		}
	}
}

void entry::onleftdown(wxMouseEvent &evt)
{
	if (!hasfocus())
		return;


	int ret = onspinner(evt.GetX(), evt.GetY());
	if (ret != 0)
	{
		if (ret>0)
			increment();
		else
			decrement();
		
		incrementing = ret > 0;
		
		if (mode == INTEGER || mode == FLOAT)
		{
			spinning = true;
			curtimer = TMRSPINNER;
			wxTimer::Start(110, false);
		}
	}
	else
	{
		mouse_down = true;
		drag_scrolling = false;
		cursor_pos = offset + xpostocursorpos(buffer+offset, evt.GetX() - this->getx() - margin);
		selection_anchor = selection_drag = cursor_pos;
		redraw();
	}

}

void entry::onleftup(wxMouseEvent &evt)
{
	if (!hasfocus())
		return;


	if (mouse_down)
	{
		drag_scrolling = false;
		mouse_down = false;
	}
	else if (spinning)
	{
		spinning = false;
	}
}
	
void entry::Notify()
{
	if (!hasfocus())
	{
		wxTimer::Stop();
		return;
	}


	if (curtimer == TMRSCROLL)
	{
		
		if (!drag_scrolling)
		{
			timer_id = -1;
		}
		else if (scrolling_left)
			cursbackward(true);
		else
			cursforward(true);
	}
	else if (curtimer == TMRSPINNER)
	{
		if (!spinning)
		{
			wxTimer::Stop();
			spintimer_id = -1;
		}
		else if (incrementing)
			increment();
		else
			decrement();
	}
}

int entry::xpostocursorpos(char *s, int xpos)
{
	char *tmp;
	int dist;
	int w = eawidth() - 2*margin;

	wxClientDC dc(gethwnd());
	Painter gfx(&dc);
	gfx.SetHelveticaSmall();
		
	if ( xpos > w )
		xpos = w;
	if ( xpos <= 0 )
	   return 0;
	
	dist = xpos;
	tmp = s;
	while ( *tmp && dist > 0 )
		dist -= gfx.GetTextWidth(tmp++, 1);
		
	if ( dist < 0 && ( xpos - dist > w || gfx.GetTextWidth(tmp-1, 1)/2 < -dist))
		tmp--;
		
	return tmp - s;
}

int entry::showlastpartoffset(char *s, int width)
{
	wxClientDC dc(gethwnd());
	Painter gfx(&dc);
	gfx.SetHelveticaSmall();

	if ( !s || s[0] == '\0' )
		return 0;
		
	char *tmp = &s[strlen( s ) - 1];
	
	do
	{
		width -= gfx.GetTextWidth( tmp--, 1 );
	}
	while ( tmp >=s && width >=0 );

	return width < 0 ? tmp - s + 2 : 0;
}

int entry::lastcharvisible()
{
	int disp_width = eawidth() - margin * 2;
	return offset + xpostocursorpos( &buffer[offset], this->getx()+disp_width);
}

int entry::selstart()
{
	return MIN(selection_anchor, selection_drag);
}

int entry::selend()
{
	return MAX(selection_anchor, selection_drag);
}

void entry::growbuffer()
{
	if (buffer_size == 0)
		buffer_size = 64;
	else 
		buffer_size *= 2;
	
	char *new_buf = new char[buffer_size];
	
	for (int i=0;i<buffer_size;i++)
		new_buf[i] = '\0';
		
	
	if (buffer)
	{
		strcpy(new_buf, buffer);
		delete [] buffer;
	}
	
	buffer = new_buf;	
}

int entry::onspinner(int nx, int ny)
{
	if (mode != STRING)
	{
		int x = nx - this->getx();
		int y = ny - this->gety();
	
		if (x >= this->getwidth() - EASPINWIDTH)
		{
			int h = getheight();
			if (y < h/2)
				return 1;
			else
				return -1;
		}
	}
	
	return 0;
}

int entry::eawidth()
{
	if (mode == STRING)
		return getwidth();
	else
		return getwidth()-EASPINWIDTH;
}

void entry::setval(int v)
{
	// the main buffer is guaranteed
	// greater than 50 chars
	char buf[50];
	mode = INTEGER;
	ivalue = v;
	
	sprintf(buf, "%d", v);
	strcpy(buffer, buf);
	text_length = strlen(buffer);
	selection_anchor = selection_drag = 0;
	if (cursor_pos > text_length)
		cursor_pos = text_length;
		
	redraw();
	
	if (changedcb != NULL)
		(*changedcb)(this, changed_data);
}

void entry::setval(float v, int digits)
{
	// the main buffer is guaranteed
	// greater than 50 chars
	char buf[50];
	char fmt[10];
	
	mode = FLOAT;
	if (digits > 0)
		ndigits = digits;
		
	fvalue = v;
	
	sprintf(fmt, "%%.%df", ndigits);
	sprintf(buf, fmt, v);
	strcpy(buffer, buf);
	text_length = strlen(buffer);
	selection_anchor = selection_drag = 0;
	if (cursor_pos > text_length)
		cursor_pos = text_length;
	
	redraw();
	
	if (changedcb != NULL)
		(*changedcb)(this, changed_data);
}

void entry::setval(bool b)
{
	mode = BOOLEAN;
	ivalue = (int)b;
	if (b)
		strcpy(buffer, "True");
	else
		strcpy(buffer, "False");
	
	text_length = strlen(buffer);
	selection_anchor = selection_drag = 0;
	if (cursor_pos > text_length)
		cursor_pos = text_length;
	
	redraw();
	

	if (changedcb != NULL)
		(*changedcb)(this, changed_data);
}

int entry::getint()
{
	return clampval(ivalue);
}

float entry::getfloat()
{
	return clampval(fvalue);
}

bool entry::getbool()
{
	return (bool)ivalue;
}

void entry::setstep(float val)
{
	step = val;
}

float entry::getstep()
{
	return step;
}

void entry::setrange(float min, float max)
{
	minimum = min;
	maximum = max;
}

void entry::getrange(float &min, float &max)
{
	min = minimum;
	max = maximum;
}

void entry::seteditable(bool b)
{
	editable = b;
}

bool entry::iseditable()
{
	return editable;
}

void entry::setmode(int i)
{
	mode = i;
	switch(mode)
	{
	case INTEGER:
		setval(ivalue);
		break;
	case FLOAT:
		setval(fvalue);
		break;
	case BOOLEAN:
		setval((bool)ivalue);
	default:
		break;
	}
}

int entry::getmode()
{
	return mode;
}

void entry::increment()
{
	switch (mode)
	{
	case BOOLEAN:
		setval( (bool) !ivalue );
		break;
	case FLOAT:
		setval(clampval(fvalue + step));
		break;
	case INTEGER:
		setval(clampval(ivalue + (int)step));
	default:
		break;
	}
}

void entry::decrement()
{
	switch (mode)
	{
	case BOOLEAN:
		setval( (bool) !ivalue );
		break;
	case FLOAT:
		setval(clampval(fvalue - step));
		break;
	case INTEGER:
		setval(clampval(ivalue - (int)step));
	default:
		break;
	}
}

bool entry::charok(char c)
{
	if (mode == INTEGER)
	{
		if (!isdigit(c) && c != '-')
			return false;
	}
	else if (mode == FLOAT)
	{
		if (!isdigit(c) && c != '.' &&
			tolower(c) != 'e' && c != '+' && c != '-')
			return false;
	}
	
	return true;
}

int entry::clampval(int v)
{
	if (v < (int)minimum)
		return (int)minimum;
	else if (v > (int)maximum)
		return (int)maximum;
	else
		return v;
}

float entry::clampval(float f)
{
	if (f < (int)minimum)
		return minimum;
	else if (f > (int)maximum)
		return maximum;
	else
		return f;
}

bool entry::isoutofrange()
{
	if (mode == INTEGER)
		return (ivalue < (int)minimum) || (ivalue > (int)maximum);
	else if (mode == FLOAT)
		return fvalue < minimum || fvalue > maximum;
	else
		return false;
}
