#include "controls.h"
#include "guibuiltin.h"
#include "xcsguiview.h"


Property::Property()
{
	object = NULL;
	sync_cb = NULL;
	ivalue = -1;
	bvalue = 0;
	type = PROP_INTEGER;
	step=min=max= -999;
}

Property::Property(const wxString &name, int type)
{
	object = NULL;
	sync_cb = NULL;
	this->type = type;
	this->name = name;
	ivalue = -1;
	bvalue = 0;
	step=min=max= -999;
}

Property &Property::operator=(const Property &p)
{
	ivalue = p.ivalue;
	fvalue = p.fvalue;
	bvalue = p.bvalue;
	string = p.string;
	string_array = p.string_array;
	menu = p.menu;
	colour = p.colour;

	return *this;
}

void Property::Write(FILE *fp)
{
	int i;
	if (!fp)
		return;
		
	fprintf(fp, "startprop %d %s\n", type, name.c_str());
	switch(type)
	{
	case PROP_STRING:
		fprintf(fp, "%s\n", string.c_str());
		break;
	case PROP_STRING_ARRAY:
		fprintf(fp, "%d\n", string_array.Count());
		for (i=0;i<(int)string_array.Count();i++)
			fprintf(fp, "%s\n", string_array[i].c_str());
		break;
	case PROP_INTEGER:
		fprintf(fp, "%d\n", ivalue);
		break;
	case PROP_FLOAT:
		fprintf(fp, "%f\n", fvalue);
		break;
	case PROP_BOOLEAN:
		fprintf(fp, "%d\n", bvalue);
		break;
	case PROP_COLOUR:
		fprintf(fp, "%d %d %d\n", colour.Red(), colour.Green(), colour.Blue());
		break;
	
	default:
		break;
	}
	
	fprintf(fp, "endprop %s\n", name.c_str());
}

static bool read_line(FILE *fp, char *buf) 
{
	if (!fgets(buf, 512, fp))
		return false;
	
	int len = strlen(buf);
	
	// hack of the '\n'
	if (len > 0)
		*(buf + len - 1) = '\0';
	
	return true;
}

bool Property::Read(FILE *fp)
{
	int type;
	int i;
	
	char propname[256];
	char linebuf[512];
	
	if (!fp)
		return false;
	
	read_line(fp, linebuf);
	sscanf(linebuf, "startprop %d %s\n", &type, propname);
	
	this->name = wxString(propname);
	this->type = type;
	
	switch(type)
	{
	case PROP_STRING:
		read_line(fp, linebuf);
		string = linebuf;
		break;
		
	case PROP_STRING_ARRAY:
	 {
		int nstrings;
		string_array.Clear();
		fscanf(fp, "%d\n", &nstrings);
		for (i=0;i<nstrings;i++)
		{
			read_line(fp, linebuf);
			string_array.Add( wxString(linebuf) );
		}
	 }
		break;
	case PROP_INTEGER:
		fscanf(fp, "%d\n", &ivalue);
		break;
		
	case PROP_FLOAT:
		fscanf(fp, "%f\n", &fvalue);
		break;
		
	case PROP_BOOLEAN:
	 {
		int x = 0;
		fscanf(fp, "%d\n", &x);
		bvalue = (bool)x;
	 }
		break;
	case PROP_COLOUR:
	 {
		int r, g, b;
		fscanf(fp, "%d %d %d\n", &r, &g, &b);
		colour = wxColour((unsigned char)r, (unsigned char)g, (unsigned char)b);
	 }
	default:
		break;
	}
		
	fscanf(fp, "endprop %s\n", linebuf);
	if (strcmp(linebuf, name.c_str()) != 0)
		wxMessageBox(wxString::Format("property read error '%s'\n", linebuf));
	
	return true;
}

/*
wxRect mRect;
wxString mName;
Array<Property> mProps;
Array<Object*> mChildren;
Dialog *mDialog;
ControlInfo *mCtrlInfo;
*/

void cb_sync_name_prop(Property *prop)
{
	if (prop && prop->object)
		prop->object->SetName( prop->string );
}

void cb_sync_geom_props(Property *prop)
{
	if (!prop || !prop->object)
		return;
	
	Object *obj = prop->object;
	wxRect rct = obj->GetGeom();

	if (prop->name == "x" && obj->GetDialog() != NULL)
		rct.x = prop->ivalue + obj->GetDialog()->GetX();
	else if (prop->name == "y" && obj->GetDialog() != NULL)
		rct.y = prop->ivalue + obj->GetDialog()->GetY();
	else if (prop->name == "width")
		rct.width = prop->ivalue;
	else if (prop->name == "height")
		rct.height = prop->ivalue;

	obj->SetGeom( rct );
	obj->Redraw();
}

Object::Object(ControlInfo *info, const wxString &name)
{
	mId = 0;
	mWxrDone = false;

	mDialog = NULL;
	mCtrlInfo = info;
	mName = name;
	mRect = wxRect(info->x, info->y, info->width, info->height);
	mProps.clear();
	mData = 0;

	// add the name property
	mNameProp = new Property("name", PROP_STRING);
	mNameProp->string = mName;
	mNameProp->object = this;
	mNameProp->sync_cb = cb_sync_name_prop;
	mProps.append( mNameProp );

	// add the geometry (x,y,width,height) properties
	mXProp = new Property("x", PROP_INTEGER);
	mXProp->ivalue = mRect.x;
	mXProp->object = this;
	mXProp->sync_cb = cb_sync_geom_props;
	mProps.append( mXProp );
	mXProp->min = 0; mXProp->max = 1000; mXProp->step = 5;

	mYProp = new Property("y", PROP_INTEGER);
	mYProp->ivalue = mRect.y;
	mYProp->object = this;
	mYProp->sync_cb = cb_sync_geom_props;
	mYProp->min = 0; mYProp->max = 1000; mYProp->step = 5;
	mProps.append( mYProp );

	mWidthProp = new Property("width", PROP_INTEGER);
	mWidthProp->ivalue = mRect.width;
	mWidthProp->object = this;
	mWidthProp->sync_cb = cb_sync_geom_props;
	mWidthProp->min = 5; mWidthProp->max = 1000; mWidthProp->step = 5;
	mProps.append( mWidthProp );

	mHeightProp = new Property("height", PROP_INTEGER);
	mHeightProp->ivalue = mRect.height;
	mHeightProp->object= this;
	mHeightProp->sync_cb = cb_sync_geom_props;
	mHeightProp->min = 5; mHeightProp->max = 1000; mHeightProp->step = 5;
	mProps.append( mHeightProp );

	// add the blank object properties
	int i=0;
	while (info->properties[i].name != NULL)
	{
		Property *p =  new Property(wxString(info->properties[i].name), info->properties[i].type);
		p->object = this;
		p->sync_cb = info->properties[i].sync;
		mProps.append( p );
		i++;
	}

	// initialize the object class
	(*info->objinit)(this, info);
}

Object::~Object()
{
	int ct = mProps.count();
	for (int i=0;i<ct;i++)
	{
		delete mProps[i];
	}
}


int Object::GetTabOrder()
{
	return GetPropInt("taborder");
}

void Object::SetId(int id)
{
	mId = id;
}

int Object::GetId()
{
	return mId;
}

void Object::SetWxrDone(bool b)
{
	mWxrDone = b;
}

bool Object::IsWxrDone()
{
	return mWxrDone;
}


ControlInfo *Object::GetControlInfo()
{
	return mCtrlInfo;
}

wxString Object::GetName()
{
	return mName;
}

void Object::SetName(const wxString &name)
{
	mName = name;
}

Property *Object::FindProp(const wxString &name)
{
	int ct = mProps.count();
	for (int i=0;i<ct;i++)
	{
		if (mProps[i]->name == name)
			return mProps[i];
	}
	return NULL;
}

Property *Object::FindOrAddProp(const wxString &name)
{
	Property *p = FindProp(name);
	if (!p)
	{
		mProps.append( p = new Property(name) );
	}

	return p;
}

int Object::NumProps()
{
	return mProps.count();
}

Property *Object::GetProp(int i)
{
	return mProps[i];
}

void Object::SetProp(const wxString &name, int val)
{
	Property *p = FindOrAddProp(name);
	p->type = PROP_INTEGER;
	p->ivalue = val;
}

void Object::SetProp(const wxString &name, bool val)
{
	Property *p = FindOrAddProp(name);
	p->type = PROP_BOOLEAN;
	p->bvalue = val;
}

void Object::SetProp(const wxString &name, float val)
{
	Property *p = FindOrAddProp(name);
	p->type = PROP_FLOAT;
	p->fvalue = val;
}

void Object::SetProp(const wxString &name, const wxString &str)
{
	Property *p = FindOrAddProp(name);
	p->type = PROP_STRING;
	p->string = str;
}

void Object::SetProp(const wxString &name, const wxArrayString &strs)
{
	Property *p = FindOrAddProp(name);
	p->type = PROP_STRING_ARRAY;
	p->string_array = strs;
}

void Object::SetProp(const wxString &name, const wxColour &colour)
{
	Property *p = FindOrAddProp(name);
	p->type = PROP_COLOUR;
	p->colour = colour;
}

void Object::SetProp(const wxString &name, const Array<MenuDef> menu)
{
	Property *p = FindOrAddProp(name);
	p->type = PROP_MENU;
	p->menu = menu;
}


int Object::GetPropInt(const wxString &name)
{
	Property *p = FindProp(name);
	if (p)
		return p->ivalue;
	else
		return -999;
}

bool Object::GetPropBool(const wxString &name)
{
	Property *p = FindProp(name);
	if (p)
		return p->bvalue;
	else
		return false;
}

float Object::GetPropFloat(const wxString &name)
{
	Property *p = FindProp(name);
	if (p)
		return p->fvalue;
	else
		return (float)-999.99;
}

wxString Object::GetPropString(const wxString &name)
{
	Property *p = FindProp(name);
	if (p)
		return p->string;
	else
		return "<property_error>";
}

wxColour Object::GetPropColour(const wxString &name)
{
	Property *p = FindProp(name);
	if (p)
		return p->colour;
	else
		return wxColour("magenta");
}


void Object::SetGeom(const wxRect &rect)
{
	SetPos(rect.x, rect.y);
	mRect.width = rect.width;
	mRect.height = rect.height;

	mWidthProp->ivalue = rect.width;
	mHeightProp->ivalue = rect.height;
		
}

void Object::SetGeom(int x, int y, int width, int height)
{
	SetPos(x, y);
	mRect.width = width;
	mRect.height = height;

	mWidthProp->ivalue = width;
	mHeightProp->ivalue = height;
}

wxRect Object::GetGeom()
{
	if (mDialog)
		return wxRect(mRect.x + mDialog->GetX(), mRect.y + mDialog->GetY(),
				mRect.width, mRect.height);
	else
		return mRect;
}

void Object::SetPos(int x, int y)
{
	mRect.x = x;
	mRect.y = y;

	if (mDialog)
	{
		mRect.x -= mDialog->GetX();
		mRect.y -= mDialog->GetY();
	}

	if (mXProp)
		mXProp->ivalue = mRect.x;
	
	if (mYProp)
		mYProp->ivalue = mRect.y;
}

int Object::GetX()
{
	if (mDialog)
		return mDialog->GetX() + mRect.x;
	else
		return mRect.x;
}

int Object::GetY()
{
	if (mDialog)
		return mDialog->GetY() + mRect.y;
	else
		return mRect.y;
}


Dialog *Object::GetDialog()
{
	return mDialog;
}

void Object::SetDialog(Dialog *obj)
{
	if (mDialog)
		mDialog->RemoveChild(this);

	mDialog = obj;

	if (mDialog)
		mDialog->AddChild(this);
}

void Object::SetData(long d)
{
	mData = d;
}

long Object::GetData()
{
	return mData;
}

void Object::Redraw()
{
	if (mDialog)
		mDialog->GetGUIView()->RedrawAll();
}


Dialog::Dialog(XcsGUIView *view, const wxString &name)
	: Object( XcsFindControlInfo("Dialog"), name)
{
	mChildren.clear();
	mGUIView = view;

	Property *p = FindProp("x");
	mProps.remove(p);
	p = FindProp("y");
	mProps.remove(p);
	mXProp = mYProp = NULL;
}

Dialog::~Dialog()
{
	/* nothing to do */
}

void Dialog::Redraw()
{
	GetGUIView()->RedrawAll();
}

wxString Dialog::GetTitle()
{
	Property *p = FindProp("title");
	if (p)
		return p->string;
	else
		return "";
}

bool Dialog::IsModal()
{
	Property *p = FindProp("modal");
	if (p)
		return p->bvalue;
	else
		return false;
}

void Dialog::AddChild(Object *obj)
{
	if (mChildren.find(obj) < 0)
		mChildren.prepend(obj);
}

void Dialog::RemoveChild(Object *obj)
{
	mChildren.remove(obj);
}

void Dialog::DeleteChildren()
{

	while (mChildren.count() > 0)
	{
		Object *child = mChildren[0];
		mChildren.remove(0);
		delete child;
	}
}

Object *Dialog::FindChild(const wxString &name)
{
	int ct = mChildren.count();
	for (int i=0;i<ct;i++)
	{
		if (mChildren[i]->GetName() == name)
			return mChildren[i];
	}
	return NULL;
}

Object** Dialog::GetChildren(int &count)
{
	count = mChildren.count();
	return mChildren.data();
}

void Dialog::RaiseChild(Object *child)
{
	int selfindex = -1;
	int i, count;

	count = mChildren.count();
	for (i=0;i<count;i++)
	{
		if (mChildren[i] == child)
		{
			selfindex = i;
			break;
		}
	}

	if (selfindex < 0)
		return;

	for (i=selfindex; i > 0; i--)
		mChildren[i] = mChildren[i-1];

	mChildren[0] = child;
}

XcsGUIView *Dialog::GetGUIView()
{
	return mGUIView;
}

