#ifndef __array_h
#define __array_h

#include <assert.h>
#include <stdlib.h>
#include <stdarg.h>

#include "vector.h"

template <class T>
class Array
{
public:
	Array(int size = 5);
	
	Array &operator= (const Array &array);
	T operator[] (int i) const;
	T &operator[] (int i);
	
	void prepend(const T &s);
	void append(const T &s);
	void insert(const T &s, int index);
	
	int find(const T &s);
	
	void remove(const T &s);
	void remove(int index);
	void clear();
	
	
	void preAlloc(int size);
	int count() const;
	int length() const;
	int capacity() const;

	static Array make(int n, ...);

	T *data();
	
private:
	Vector<T> items;
	int n_items;
};
	


////////////////////////////////////////////////////////////
// templated implementation

template <class T>
Array<T>::Array(int size)
{
	n_items = 0;
	if (size > 0)
		items.resize(size);
}

template <class T>
Array<T> &Array<T>::operator= (const Array &array)
{
	if (this != &array)
	{
		items.resize(array.length());
		n_items = array.length();
		
		for (int i=0;i<items.length();i++)
			items[i] = array.items[i];
	}

	return *this;
}

template <class T>
T Array<T>::operator[] (int i) const
{
	assert( i >= 0 && i < n_items );
	
	return items[i];
}

template <class T>
T &Array<T>::operator[] (int i)
{
	assert( i >= 0 && i < n_items );
	
	return items[i];
}

template <class T>
void Array<T>::prepend(const T &s)
{
	insert(s, 0);
}

template <class T>
void Array<T>::append(const T &s)
{
	int last_location = n_items;
	
	if (items.length() < n_items + 1)
		items.resize(items.length() + 10);
	
	items[last_location] = s;
	n_items++;
}

template <class T>
void Array<T>::insert(const T &s, int index)
{
	assert( index >=0 && index <= items.length() );
		
	if (items.length() < n_items + 1)
		items.resize(items.length() + 10);
		
	for (int i=n_items; i > index; i--)
		items[i] = items[i-1];
	
	items[index] = s;
	n_items++;
}

template <class T>
int Array<T>::find(const T &s)
{
	for (int i=0;i<n_items;i++)
		if (s == items[i])
			return i;
	
	return -1;
}

template <class T>
void Array<T>::remove(const T &s)
{
	int index = find(s);
	if (index >= 0)
		remove(index);
}

template <class T>
void Array<T>::remove(int index)
{
	assert(index >= 0 && index < n_items);
	
	for (int i=index;i<n_items-1;i++)
		items[i] = items[i+1];
	
	n_items--;
}

template <class T>
void Array<T>::clear()
{
	n_items = 0;
}

template <class T>
void Array<T>::preAlloc(int size)
{
	if (size > items.length())
		items.resize(size);
}

template <class T>
int Array<T>::length() const
{
	return n_items;
}

template <class T>
int Array<T>::count() const
{
	return n_items;
}

template <class T>
int Array<T>::capacity() const
{
	return items.length();
}

template <class T>
Array<T> Array<T>::make(int n, ...)
{
	char *str;
	Array sa;
	va_list args;
	
	va_start(args, n);

	for (int i=0;i<n;i++)
	{
		str = va_arg(args, char*);
		sa.append( str );	  
	}
	
	va_end(args);
	
	return sa;
}

template <class T>
T *Array<T>::data()
{
	return items.data();
}

#endif
